/*
 * Decompiled with CFR 0.152.
 */
package org.opensha.commons.geo;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Range;
import com.google.common.primitives.Doubles;
import com.google.gson.TypeAdapter;
import com.google.gson.annotations.JsonAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;
import java.awt.geom.Area;
import java.awt.geom.Rectangle2D;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.math3.util.Precision;
import org.dom4j.Element;
import org.opensha.commons.data.region.CaliforniaRegions;
import org.opensha.commons.geo.BorderType;
import org.opensha.commons.geo.Direction;
import org.opensha.commons.geo.Location;
import org.opensha.commons.geo.LocationList;
import org.opensha.commons.geo.Region;
import org.opensha.commons.geo.RegionUtils;
import org.opensha.commons.geo.json.Feature;
import org.opensha.commons.geo.json.FeatureProperties;
import org.opensha.commons.geo.json.Geometry;

@JsonAdapter(value=Adapter.class)
public class GriddedRegion
extends Region
implements Iterable<Location> {
    private static final long serialVersionUID = 1L;
    public static final String XML_METADATA_NAME = "evenlyGriddedGeographicRegion";
    public static final String XML_METADATA_GRID_SPACING_NAME = "spacing";
    public static final String XML_METADATA_ANCHOR_NAME = "anchor";
    public static final String XML_METADATA_NUM_POINTS_NAME = "numPoints";
    public static final Location ANCHOR_0_0 = new Location(0.0, 0.0);
    private double minGridLat;
    private double minGridLon;
    private double maxGridLat;
    private double maxGridLon;
    private int numLonNodes;
    private int numLatNodes;
    private double[] lonNodeCenters;
    private double[] latNodeCenters;
    private Location anchor;
    private int[] gridIndices;
    private LocationList nodeList;
    private double latSpacing;
    private double lonSpacing;
    private int nodeCount;
    private static final double PRECISION_SCALE = 1.00000000000001;
    private static final String JSON_LAT_NODES = "LatNodes";
    private static final String JSON_LON_NODES = "LonNodes";
    private static final String JSON_LAT_SPACING = "LatSpacing";
    private static final String JSON_LON_SPACING = "LonSpacing";
    private static final String JSON_ANCHOR = "Anchor";

    public GriddedRegion(Location loc1, Location loc2, double latSpacing, double lonSpacing, Location anchor) {
        super(loc1, loc2);
        this.initGrid(latSpacing, lonSpacing, anchor);
    }

    public GriddedRegion(Location loc1, Location loc2, double spacing, Location anchor) {
        this(loc1, loc2, spacing, spacing, anchor);
    }

    public GriddedRegion(LocationList border, BorderType type, double latSpacing, double lonSpacing, Location anchor) {
        super(border, type);
        this.initGrid(latSpacing, lonSpacing, anchor);
    }

    public GriddedRegion(LocationList border, BorderType type, double spacing, Location anchor) {
        this(border, type, spacing, spacing, anchor);
    }

    public GriddedRegion(Location center, double radius, double latSpacing, double lonSpacing, Location anchor) {
        super(center, radius);
        this.initGrid(latSpacing, lonSpacing, anchor);
    }

    public GriddedRegion(Location center, double radius, double spacing, Location anchor) {
        this(center, radius, spacing, spacing, anchor);
    }

    public GriddedRegion(LocationList line, double buffer, double latSpacing, double lonSpacing, Location anchor) {
        super(line, buffer);
        this.initGrid(latSpacing, lonSpacing, anchor);
    }

    public GriddedRegion(LocationList line, double buffer, double spacing, Location anchor) {
        this(line, buffer, spacing, spacing, anchor);
    }

    public GriddedRegion(Region region, double latSpacing, double lonSpacing, Location anchor) {
        super(region);
        this.initGrid(latSpacing, lonSpacing, anchor);
    }

    public GriddedRegion(Region region, double spacing, Location anchor) {
        this(region, spacing, spacing, anchor);
    }

    public GriddedRegion(GriddedRegion gridRegion) {
        this(gridRegion, gridRegion.latNodeCenters, gridRegion.lonNodeCenters, gridRegion.latSpacing, gridRegion.lonSpacing, gridRegion.anchor, gridRegion.nodeList);
    }

    private GriddedRegion(Region region, double[] latNodeCenters, double[] lonNodeCenters, double latSpacing, double lonSpacing, Location anchor, LocationList nodeList) {
        super(region);
        this.latNodeCenters = latNodeCenters;
        this.lonNodeCenters = lonNodeCenters;
        this.setSpacing(latSpacing, lonSpacing);
        if (anchor == null) {
            this.setAnchor(null);
        } else {
            this.anchor = anchor;
        }
        this.initGridRange();
        int gridSize = this.numLonNodes * this.numLatNodes;
        this.nodeList = nodeList;
        this.nodeCount = nodeList.size();
        this.gridIndices = new int[gridSize];
        for (int i = 0; i < gridSize; ++i) {
            this.gridIndices[i] = -1;
        }
        int node_idx = 0;
        while (node_idx < this.nodeCount) {
            Location node = (Location)nodeList.get(node_idx);
            int latIndex = this.getLatIndex(node);
            Preconditions.checkState((latIndex >= 0 ? 1 : 0) != 0, (String)"Couldn't find latitude index for grid node %s: %s", (int)node_idx, (Object)node);
            int lonIndex = this.getLonIndex(node);
            Preconditions.checkState((lonIndex >= 0 ? 1 : 0) != 0, (String)"Couldn't find longitude index for grid node %s: %s", (int)node_idx, (Object)node);
            int grid_idx = latIndex * this.numLonNodes + lonIndex;
            Preconditions.checkState((this.gridIndices[grid_idx] == -1 ? 1 : 0) != 0, (Object)"Multiple nodes map to the same grid index");
            this.gridIndices[grid_idx] = node_idx++;
        }
    }

    public double getLatSpacing() {
        return this.latSpacing;
    }

    public double getLonSpacing() {
        return this.lonSpacing;
    }

    public double getSpacing() {
        return this.latSpacing;
    }

    public boolean isSpacingUniform() {
        return this.latSpacing == this.lonSpacing;
    }

    public int getNodeCount() {
        return this.nodeCount;
    }

    public int getNumLocations() {
        return this.getNodeCount();
    }

    public boolean isEmpty() {
        return this.nodeCount == 0;
    }

    public int move(int idx, Direction dir) {
        Location start = this.locationForIndex(idx);
        Preconditions.checkNotNull((Object)start, (Object)"Invalid start index");
        Location end = new Location(start.getLatitude() + this.latSpacing * (double)dir.signLatMove(), start.getLongitude() + this.lonSpacing * (double)dir.signLonMove());
        return this.indexForLocation(end);
    }

    public boolean equalsRegion(GriddedRegion gr) {
        if (!super.equalsRegion(gr)) {
            return false;
        }
        if (!gr.anchor.equals(this.anchor)) {
            return false;
        }
        if (gr.latSpacing != this.latSpacing) {
            return false;
        }
        return gr.lonSpacing == this.lonSpacing;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof GriddedRegion)) {
            return false;
        }
        GriddedRegion gr = (GriddedRegion)obj;
        if (!this.getName().equals(gr.getName())) {
            return false;
        }
        return this.equalsRegion(gr);
    }

    @Override
    public int hashCode() {
        return super.hashCode() ^ this.anchor.hashCode() ^ Double.valueOf(this.latSpacing).hashCode() ^ Double.valueOf(this.lonSpacing).hashCode();
    }

    @Override
    public GriddedRegion clone() {
        return new GriddedRegion(this);
    }

    public GriddedRegion subRegion(Region region) {
        Region newRegion = Region.intersect(this, region);
        if (newRegion == null) {
            return null;
        }
        return new GriddedRegion(newRegion, this.latSpacing, this.lonSpacing, this.anchor);
    }

    @Override
    public void addInterior(Region region) {
        throw new UnsupportedOperationException("A GriddedRegion may not have an interior Region set");
    }

    @Override
    public Iterator<Location> iterator() {
        return this.nodeList.iterator();
    }

    public LocationList getNodeList() {
        return this.nodeList;
    }

    public Location locationForIndex(int index) {
        try {
            return (Location)this.nodeList.get(index);
        }
        catch (IndexOutOfBoundsException e) {
            return null;
        }
    }

    public List<Integer> indicesForBounds(Rectangle2D rect) {
        try {
            return this.indexLookupFast(rect);
        }
        catch (Exception e) {
            return this.indexLookupSlow(rect);
        }
    }

    private List<Integer> indexLookupFast(Rectangle2D rect) {
        Preconditions.checkArgument((boolean)this.area.contains(rect));
        Location pLL = new Location(rect.getMinY(), rect.getMinX());
        Location pUL = new Location(rect.getMaxY(), rect.getMinX());
        Location pLR = new Location(rect.getMinY(), rect.getMaxX());
        int idxLL = this.indexForLocation(pLL);
        int idxUL = this.indexForLocation(pUL);
        int idxLR = this.indexForLocation(pLR);
        ArrayList rowStarts = Lists.newArrayList();
        int rowStartIdx = idxLL;
        int lastRowStartIdx = idxUL;
        while (rowStartIdx <= lastRowStartIdx) {
            rowStarts.add(rowStartIdx);
            Location currLoc = this.locationForIndex(rowStartIdx);
            Location nextLoc = new Location(currLoc.getLatitude() + this.latSpacing, currLoc.getLongitude());
            rowStartIdx = this.indexForLocation(nextLoc);
        }
        int len = idxLR - idxLL + 1;
        ArrayList indices = Lists.newArrayList();
        for (Integer idx : rowStarts) {
            GriddedRegion.addRange(indices, idx, len);
        }
        return indices;
    }

    private List<Integer> indexLookupSlow(Rectangle2D rect) {
        ArrayList indices = Lists.newArrayList();
        for (int i = 0; i < this.nodeCount; ++i) {
            Area area = this.areaForIndex(i);
            if (!area.intersects(rect)) continue;
            indices.add(i);
        }
        return indices;
    }

    private static void addRange(List<Integer> ints, int start, int num) {
        for (int i = start; i < start + num; ++i) {
            ints.add(i);
        }
    }

    public Area areaForIndex(int index) {
        Location p = this.locationForIndex(index);
        return RegionUtils.getNodeShape(p, this.lonSpacing, this.latSpacing);
    }

    public Location getLocation(int index) {
        return this.locationForIndex(index);
    }

    public int indexForLocation(Location loc) {
        int lonIndex = GriddedRegion.getNodeIndex(this.lonNodeCenters, loc.getLongitude(), this.lonSpacing);
        if (lonIndex == -1) {
            return -1;
        }
        int latIndex = GriddedRegion.getNodeIndex(this.latNodeCenters, loc.getLatitude(), this.latSpacing);
        if (latIndex == -1) {
            return -1;
        }
        int gridIndex = latIndex * this.numLonNodes + lonIndex;
        return this.gridIndices[gridIndex];
    }

    public int getLatIndex(Location loc) {
        return GriddedRegion.getNodeIndex(this.latNodeCenters, loc.getLatitude(), this.latSpacing);
    }

    public int getLonIndex(Location loc) {
        return GriddedRegion.getNodeIndex(this.lonNodeCenters, loc.getLongitude(), this.lonSpacing);
    }

    public double[] getLatNodes() {
        return Arrays.copyOf(this.latNodeCenters, this.latNodeCenters.length);
    }

    public double[] getLonNodes() {
        return Arrays.copyOf(this.lonNodeCenters, this.lonNodeCenters.length);
    }

    public int getNodeIndex(int latIndex, int lonIndex) {
        if (latIndex == -1 || latIndex == -1) {
            return -1;
        }
        return latIndex * this.numLonNodes + lonIndex;
    }

    public int getNumLatNodes() {
        return this.numLatNodes;
    }

    public int getNumLonNodes() {
        return this.numLonNodes;
    }

    public double getMinGridLat() {
        return this.minGridLat;
    }

    public double getMaxGridLat() {
        return this.maxGridLat;
    }

    public double getMinGridLon() {
        return this.minGridLon;
    }

    public double getMaxGridLon() {
        return this.maxGridLon;
    }

    @Override
    public Element toXMLMetadata(Element root) {
        Element xml = root.addElement(XML_METADATA_NAME);
        xml.addAttribute(XML_METADATA_GRID_SPACING_NAME, "" + this.getSpacing());
        Element xml_anchor = xml.addElement(XML_METADATA_ANCHOR_NAME);
        this.anchor.toXMLMetadata(xml_anchor);
        xml.addAttribute(XML_METADATA_NUM_POINTS_NAME, "" + this.getNodeCount());
        super.toXMLMetadata(xml);
        return root;
    }

    public static GriddedRegion fromXMLMetadata(Element root) {
        double gridSpacing = Double.parseDouble(root.attribute(XML_METADATA_GRID_SPACING_NAME).getValue());
        Region geoRegion = Region.fromXMLMetadata(root.element("Region"));
        LocationList outline = geoRegion.getBorder();
        Location xml_anchor = Location.fromXMLMetadata(root.element(XML_METADATA_ANCHOR_NAME).element("Location"));
        return new GriddedRegion(outline, BorderType.MERCATOR_LINEAR, gridSpacing, gridSpacing, xml_anchor);
    }

    private static int getNodeIndex(double[] nodes, double value, double spacing) {
        double iVal = 1.00000000000001 * (value - nodes[0]) / spacing;
        int i = (int)Math.round(iVal);
        if (i == -1 && (float)(nodes[0] - 0.5 * spacing) == (float)value) {
            return 0;
        }
        return i < 0 ? -1 : (i >= nodes.length ? -1 : i);
    }

    private void initGrid(double latSpacing, double lonSpacing, Location anchor) {
        this.setSpacing(latSpacing, lonSpacing);
        this.setAnchor(anchor);
        this.initNodes();
    }

    private void setSpacing(double latSpacing, double lonSpacing) {
        String mssg = "spacing [%s] must be 0\u00b0 > S \u2265 5\u00b0";
        Preconditions.checkArgument((latSpacing > 0.0 && latSpacing <= 5.0 ? 1 : 0) != 0, (String)("Latitude" + mssg), (Object)latSpacing);
        Preconditions.checkArgument((lonSpacing > 0.0 && lonSpacing <= 5.0 ? 1 : 0) != 0, (String)("Longitude" + mssg), (Object)lonSpacing);
        this.latSpacing = latSpacing;
        this.lonSpacing = lonSpacing;
    }

    private void setAnchor(Location anchor) {
        if (anchor == null) {
            anchor = new Location(this.getMinLat(), this.getMinLon());
        }
        double newLat = GriddedRegion.computeAnchor(this.getMinLat(), anchor.getLatitude(), this.latSpacing);
        double newLon = GriddedRegion.computeAnchor(this.getMinLon(), anchor.getLongitude(), this.lonSpacing);
        this.anchor = new Location(newLat, newLon);
    }

    private static double computeAnchor(double min, double anchor, double spacing) {
        double delta = anchor - min;
        double num_div = Math.floor(delta / spacing);
        double offset = delta - num_div * spacing;
        double newAnchor = min + offset;
        newAnchor = newAnchor < min ? newAnchor + spacing : newAnchor;
        return Precision.round((double)newAnchor, (int)8);
    }

    private void initNodes() {
        this.lonNodeCenters = GriddedRegion.initNodeCenters(this.anchor.getLongitude(), this.getMaxLon(), this.lonSpacing);
        this.latNodeCenters = GriddedRegion.initNodeCenters(this.anchor.getLatitude(), this.getMaxLat(), this.latSpacing);
        this.initGridRange();
        int gridSize = this.numLonNodes * this.numLatNodes;
        this.gridIndices = new int[gridSize];
        this.nodeList = new LocationList();
        int node_idx = 0;
        int grid_idx = 0;
        for (double lat : this.latNodeCenters) {
            for (double lon : this.lonNodeCenters) {
                Location loc = new Location(lat, lon, 0.0);
                if (this.contains(loc)) {
                    this.nodeList.add(loc);
                    this.gridIndices[grid_idx] = node_idx++;
                } else {
                    this.gridIndices[grid_idx] = -1;
                }
                ++grid_idx;
            }
        }
        this.nodeCount = node_idx;
    }

    private void initGridRange() {
        this.numLatNodes = this.latNodeCenters.length;
        this.numLonNodes = this.lonNodeCenters.length;
        this.minGridLat = this.numLatNodes != 0 ? this.latNodeCenters[0] : Double.NaN;
        this.maxGridLat = this.numLatNodes != 0 ? this.latNodeCenters[this.numLatNodes - 1] : Double.NaN;
        this.minGridLon = this.numLonNodes != 0 ? this.lonNodeCenters[0] : Double.NaN;
        this.maxGridLon = this.numLonNodes != 0 ? this.lonNodeCenters[this.numLonNodes - 1] : Double.NaN;
    }

    private static double[] initNodeCenters(double min, double max, double width) {
        int nodeCount = (int)Math.floor((max - min) / width) + 1;
        double firstCenterVal = min;
        return GriddedRegion.buildArray(firstCenterVal, nodeCount, width);
    }

    private static double[] buildArray(double startVal, int count, double interval) {
        double[] values = new double[count];
        double val = startVal;
        for (int i = 0; i < count; ++i) {
            values[i] = Precision.round((double)val, (int)8);
            val += interval;
        }
        return values;
    }

    @Override
    public Feature toFeature() {
        ArrayList<Geometry> geometries = new ArrayList<Geometry>();
        geometries.add(new Geometry.Polygon(this));
        geometries.add(new Geometry.MultiPoint(this.nodeList));
        Geometry.GeometryCollection geometry = new Geometry.GeometryCollection(geometries);
        FeatureProperties properties = new FeatureProperties();
        properties.put(JSON_LAT_NODES, this.latNodeCenters);
        properties.put(JSON_LON_NODES, this.lonNodeCenters);
        properties.put(JSON_LAT_SPACING, this.latSpacing);
        properties.put(JSON_LON_SPACING, this.lonSpacing);
        properties.put(JSON_ANCHOR, this.anchor);
        String name = this.getName();
        if (name != null && name.equals("Unnamed Region")) {
            name = null;
        }
        return new Feature(name, (Geometry)geometry, properties);
    }

    public static GriddedRegion fromFeature(Feature feature) {
        Preconditions.checkNotNull((Object)feature.geometry, (Object)"Feature is missing geometry");
        Preconditions.checkState((boolean)(feature.geometry instanceof Geometry.GeometryCollection), (String)"Unexpected geometry type for GriddedRegion, should be GeometryCollection: %s", (Object)((Object)feature.geometry.type));
        Geometry.GeometryCollection geometries = (Geometry.GeometryCollection)feature.geometry;
        Region region = null;
        LocationList nodeList = null;
        for (Geometry geometry : geometries.geometries) {
            if (geometry instanceof Geometry.Polygon) {
                Preconditions.checkState((region == null ? 1 : 0) != 0, (Object)"Multiple region polygons found for GriddedRegion");
                region = ((Geometry.Polygon)geometry).asRegion();
                continue;
            }
            if (geometry instanceof Geometry.MultiPolygon) {
                Preconditions.checkState((region == null ? 1 : 0) != 0, (Object)"Multiple region polygons found for GriddedRegion");
                List<Region> list = ((Geometry.MultiPolygon)feature.geometry).asRegions();
                Preconditions.checkState((list.size() == 1 ? 1 : 0) != 0, (String)"Must have exactly 1 polygon, have %s", (int)list.size());
                region = list.get(0);
                continue;
            }
            if (geometry instanceof Geometry.MultiPoint) {
                Preconditions.checkState((nodeList == null ? 1 : 0) != 0, (Object)"Multiple node lists found");
                nodeList = ((Geometry.MultiPoint)geometry).points;
                continue;
            }
            if (geometry instanceof Geometry.Point) {
                Preconditions.checkState((nodeList == null ? 1 : 0) != 0, (Object)"Multiple node lists found");
                nodeList = new LocationList();
                nodeList.add(((Geometry.Point)geometry).point);
                continue;
            }
            System.err.println("Warning: skipping unexpected geometry type when loading GriddedRegion: " + String.valueOf((Object)geometry.type));
        }
        Preconditions.checkNotNull(region, (Object)"Region polygon not found in GriddedRegion feature");
        Preconditions.checkNotNull(nodeList, (Object)"Node list (MultiPoint geometry) not found in GriddedRegion feature");
        FeatureProperties properties = feature.properties;
        double[] latNodeCenters = null;
        double[] lonNodeCenters = null;
        double latSpacing = Double.NaN;
        double lonSpacing = Double.NaN;
        Location anchor = null;
        if (properties != null) {
            latNodeCenters = properties.getDoubleArray(JSON_LAT_NODES);
            lonNodeCenters = properties.getDoubleArray(JSON_LON_NODES);
            latSpacing = properties.getDouble(JSON_LAT_SPACING, Double.NaN);
            lonSpacing = properties.getDouble(JSON_LON_SPACING, Double.NaN);
            anchor = properties.getLocation(JSON_ANCHOR, null);
        }
        if (latNodeCenters == null) {
            System.err.println("Warning: LatNodes not specified in GriddedRegion GeoJSON properties, inferring from node list");
            latNodeCenters = GriddedRegion.inferNodeCenters(nodeList, true);
        }
        if (lonNodeCenters == null) {
            System.err.println("Warning: LonNodes not specified in GriddedRegion GeoJSON properties, inferring from node list");
            lonNodeCenters = GriddedRegion.inferNodeCenters(nodeList, false);
        }
        if (Double.isNaN(latSpacing)) {
            System.err.println("Warning: LatSpacing not specified in GriddedRegion GeoJSON properties, inferring from nodes");
            latSpacing = GriddedRegion.inferSpacing(latNodeCenters, false);
        }
        if (Double.isNaN(lonSpacing)) {
            System.err.println("Warning: LonSpacing not specified in GriddedRegion GeoJSON properties, inferring from nodes");
            lonSpacing = GriddedRegion.inferSpacing(lonNodeCenters, false);
        }
        GriddedRegion gridRegion = new GriddedRegion(region, latNodeCenters, lonNodeCenters, latSpacing, lonSpacing, anchor, nodeList);
        if (feature.id != null) {
            gridRegion.setName(feature.id.toString());
        }
        return gridRegion;
    }

    private static double[] inferNodeCenters(LocationList gridNodes, boolean latitude) {
        if (gridNodes.isEmpty()) {
            return new double[0];
        }
        ArrayList<Double> values = new ArrayList<Double>();
        HashSet<Float> uniques = new HashSet<Float>();
        for (Location loc : gridNodes) {
            double val = latitude ? loc.getLatitude() : loc.getLongitude();
            if (uniques.contains(Float.valueOf((float)val))) continue;
            uniques.add(Float.valueOf((float)val));
            values.add(val);
        }
        Collections.sort(values);
        double[] array = Doubles.toArray(values);
        GriddedRegion.inferSpacing(array, true);
        return array;
    }

    private static double inferSpacing(double[] values, boolean allowHoles) {
        double spacing;
        block6: {
            block5: {
                double mySpacing;
                int i;
                if (values.length < 1) {
                    return 0.0;
                }
                if (!allowHoles) break block5;
                spacing = Double.POSITIVE_INFINITY;
                boolean allEqual = true;
                for (i = 1; i < values.length; ++i) {
                    mySpacing = Math.abs(values[i] - values[i - 1]);
                    Preconditions.checkState((Double.isFinite(mySpacing) && mySpacing > 0.0 ? 1 : 0) != 0, (String)"Spacing between consecutive grid nodes must be >0 and finite: %s", (Object)mySpacing);
                    if (i == 1) {
                        spacing = mySpacing;
                        continue;
                    }
                    if (allEqual &= (float)mySpacing == (float)spacing) continue;
                    spacing = Math.min(spacing, mySpacing);
                }
                if (allEqual) break block6;
                for (i = 1; i < values.length; ++i) {
                    mySpacing = Math.abs(values[i] - values[i - 1]);
                    if ((float)mySpacing == (float)spacing) continue;
                    double multiple = mySpacing / spacing;
                    Preconditions.checkState(((float)multiple == (float)Math.round(multiple) ? 1 : 0) != 0, (String)"Does list contains holes (or is irregular). Spacing should be an exact multiple of the smallest spacing (%s), but spacing between elements %s and %s is %s.", (Object)Float.valueOf((float)spacing), (Object)(i - 1), (Object)i, (Object)Float.valueOf((float)mySpacing));
                }
                break block6;
            }
            spacing = Math.abs(values[values.length - 1] - values[0]) / (double)(values.length - 1);
            for (int i = 1; i < values.length; ++i) {
                float calcSpacing = (float)Math.abs(values[i] - values[i - 1]);
                Preconditions.checkState((calcSpacing == (float)spacing ? 1 : 0) != 0, (String)"Cannot infer spacing. Implied spacing from whole node array is %s, but spacing between elements %s and %s is %s", (Object)Float.valueOf((float)spacing), (Object)(i - 1), (Object)i, (Object)Float.valueOf(calcSpacing));
            }
        }
        return spacing;
    }

    public static double inferLatSpacing(LocationList gridLocs) {
        double[] centers = GriddedRegion.inferNodeCenters(gridLocs, true);
        return GriddedRegion.inferSpacing(centers, true);
    }

    public static double inferLonSpacing(LocationList gridLocs) {
        double[] centers = GriddedRegion.inferNodeCenters(gridLocs, false);
        return GriddedRegion.inferSpacing(centers, true);
    }

    public static GriddedRegion inferRegion(LocationList nodeList) throws IllegalStateException {
        Region region;
        double[] latNodes = GriddedRegion.inferNodeCenters(nodeList, true);
        double[] lonNodes = GriddedRegion.inferNodeCenters(nodeList, false);
        double latSpacing = GriddedRegion.inferSpacing(latNodes, false);
        double lonSpacing = GriddedRegion.inferSpacing(lonNodes, false);
        double latBuffer = latSpacing * 0.25;
        double lonBuffer = lonSpacing * 0.25;
        if (latNodes.length * lonNodes.length == nodeList.size()) {
            region = new Region(new Location(latNodes[0] - latBuffer, lonNodes[0] - lonBuffer), new Location(latNodes[latNodes.length - 1] + latBuffer, lonNodes[lonNodes.length - 1] + lonBuffer));
        } else {
            double lon;
            Range<Double> lonRange;
            double lat;
            int i;
            LocationList border = new LocationList();
            for (i = 0; i < latNodes.length; ++i) {
                lat = latNodes[i];
                lonRange = GriddedRegion.lonRangeAtLat(nodeList, (float)lat);
                lon = (Double)lonRange.lowerEndpoint() - lonBuffer;
                if (i == 0) {
                    border.add(new Location(lat - latBuffer, lon));
                    continue;
                }
                if (i == latNodes.length - 1) {
                    border.add(new Location(lat + latBuffer, lon));
                    continue;
                }
                border.add(new Location(lat, lon));
            }
            i = latNodes.length;
            while (--i >= 0) {
                lat = latNodes[i];
                lonRange = GriddedRegion.lonRangeAtLat(nodeList, (float)lat);
                lon = (Double)lonRange.upperEndpoint() + lonBuffer;
                if (i == latNodes.length - 1) {
                    border.add(new Location(lat + latBuffer, lon));
                    continue;
                }
                if (i == 0) {
                    border.add(new Location(lat - latBuffer, lon));
                    continue;
                }
                border.add(new Location(lat, lon));
            }
            region = new Region(border, BorderType.MERCATOR_LINEAR);
        }
        return new GriddedRegion(region, latNodes, lonNodes, latSpacing, lonSpacing, (Location)nodeList.get(0), nodeList);
    }

    private static Range<Double> lonRangeAtLat(LocationList nodeList, float lat) {
        double min = Double.POSITIVE_INFINITY;
        double max = Double.NEGATIVE_INFINITY;
        for (Location loc : nodeList) {
            if ((float)loc.getLatitude() != lat) continue;
            double lon = loc.getLongitude();
            max = Double.max(max, lon);
            min = Double.min(min, lon);
        }
        Preconditions.checkState((Double.isFinite(min) && Double.isFinite(max) ? 1 : 0) != 0, (String)"No nodes found at lat=%s", (Object)Float.valueOf(lat));
        return Range.closed((Comparable)Double.valueOf(min), (Comparable)Double.valueOf(max));
    }

    public static GriddedRegion inferEncompassingRegion(LocationList nodeList) {
        Location anchor = (Location)nodeList.get(0);
        double latSpacing = GriddedRegion.inferLatSpacing(nodeList);
        double lonSpacing = GriddedRegion.inferLonSpacing(nodeList);
        double minLat = Double.POSITIVE_INFINITY;
        double maxLat = Double.NEGATIVE_INFINITY;
        double minLon = Double.POSITIVE_INFINITY;
        double maxLon = Double.NEGATIVE_INFINITY;
        for (Location loc : nodeList) {
            minLat = Math.min(minLat, loc.lat);
            maxLat = Math.max(maxLat, loc.lat);
            minLon = Math.min(minLon, loc.lon);
            maxLon = Math.max(maxLon, loc.lon);
        }
        minLat = Math.max(-90.0, minLat - 0.5 * latSpacing);
        maxLat = Math.min(90.0, maxLat + 0.5 * latSpacing);
        if (minLon < 0.0) {
            Preconditions.checkState((minLon >= -180.0 ? 1 : 0) != 0);
            Preconditions.checkState((maxLon <= 180.0 ? 1 : 0) != 0);
            minLon = Math.max(-180.0, minLon - 0.5 * lonSpacing);
            maxLon = Math.min(180.0, maxLon + 0.5 * lonSpacing);
        } else {
            Preconditions.checkState((maxLon <= 360.0 ? 1 : 0) != 0);
            minLon = Math.max(0.0, minLon - 0.5 * lonSpacing);
            maxLon = Math.min(360.0, maxLon + 0.5 * lonSpacing);
        }
        Region encompassing = new Region(new Location(minLat, minLon), new Location(maxLat, maxLon));
        GriddedRegion encompassingRegion = new GriddedRegion(encompassing, latSpacing, lonSpacing, anchor);
        for (Location loc : nodeList) {
            int index = encompassingRegion.indexForLocation(loc);
            Preconditions.checkState((index >= 0 ? 1 : 0) != 0, (String)"Failed to build an encompassing region for the give node list. This location is in the original list, but doesn't map to a grid node: %s", (Object)loc);
        }
        return encompassingRegion;
    }

    public static void main(String[] args) throws IOException {
        ArrayList<GriddedRegion> inputRegions = new ArrayList<GriddedRegion>();
        GriddedRegion simple = new GriddedRegion(new Region(new Location(34.0, -118.0), new Location(36.0, -120.0)), 0.25, null);
        simple.setName("rectangular");
        inputRegions.add(simple);
        GriddedRegion circular = new GriddedRegion(new Region(new Location(35.0, -119.0), 100.0), 0.25, null);
        circular.setName("circular");
        inputRegions.add(circular);
        CaliforniaRegions.RELM_TESTING_GRIDDED relm = new CaliforniaRegions.RELM_TESTING_GRIDDED(0.25);
        relm.setName("relm");
        inputRegions.add(relm);
        File outputDir = new File("/tmp/grid_reg_test");
        Preconditions.checkState((outputDir.exists() || outputDir.mkdir() ? 1 : 0) != 0);
        for (GriddedRegion input : inputRegions) {
            System.out.println("Processing " + input.getName());
            Feature.write(input.toFeature(), new File(outputDir, input.getName() + "_input.geojson"));
            GriddedRegion inferred = GriddedRegion.inferRegion(input.getNodeList());
            Feature.write(inferred.toFeature(), new File(outputDir, input.getName() + "_inferred.geojson"));
        }
    }

    public static class Adapter
    extends TypeAdapter<GriddedRegion> {
        private Feature.FeatureAdapter featureAdapter = new Feature.FeatureAdapter();

        public void write(JsonWriter out, GriddedRegion value) throws IOException {
            if (value == null) {
                out.nullValue();
            } else {
                this.featureAdapter.write(out, value.toFeature());
            }
        }

        public GriddedRegion read(JsonReader in) throws IOException {
            Feature feature = this.featureAdapter.read(in);
            return GriddedRegion.fromFeature(feature);
        }
    }
}

