/*
 * Decompiled with CFR 0.152.
 */
package org.opensha.commons.util.io.archive;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.text.DecimalFormat;
import java.util.ArrayDeque;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.opensha.commons.data.Named;
import org.opensha.commons.util.ExecutorUtils;
import org.opensha.commons.util.io.archive.ArchiveInput;
import org.opensha.commons.util.io.archive.AsynchronousApacheZipper;
import org.opensha.commons.util.io.archive.CopyAvoidantInMemorySeekableByteChannel;

public interface ArchiveOutput
extends Closeable,
Named {
    public static final char SEPERATOR_CHAR = '/';
    public static final String SEPERATOR = "/";

    public void putNextEntry(String var1) throws IOException;

    public OutputStream getOutputStream() throws IOException;

    public void closeEntry() throws IOException;

    public ArchiveInput getCompletedInput() throws IOException;

    default public void transferFrom(ArchiveInput input, String name) throws IOException {
        this.transferFrom(input, name, name);
    }

    default public void transferFrom(ArchiveInput input, String sourceName, String destName) throws IOException {
        this.transferFrom(input.getInputStream(sourceName), destName);
    }

    default public void transferFrom(InputStream is, String name) throws IOException {
        this.putNextEntry(name);
        OutputStream os = this.getOutputStream();
        is.transferTo(os);
        os.flush();
        this.closeEntry();
    }

    public static File getDefaultInProgressFile(File outputFile) {
        return new File(outputFile.getParentFile(), outputFile.getName() + ".tmp");
    }

    public static ArchiveOutput getDefaultOutput(File outputFile) throws IOException {
        return ArchiveOutput.getDefaultOutput(outputFile, null);
    }

    public static ArchiveOutput getDefaultOutput(File outputFile, ArchiveInput input) throws IOException {
        String name = outputFile.getName().toLowerCase();
        if (outputFile.isDirectory() || !outputFile.exists() && name.endsWith(File.separator)) {
            return new DirectoryOutput(outputFile.toPath());
        }
        if (name.endsWith(".tar")) {
            return new TarFileOutput(outputFile);
        }
        if (name.endsWith(".zip")) {
            if (input instanceof ArchiveInput.AbstractApacheZipInput) {
                return new ApacheZipFileOutput(outputFile);
            }
            if (input instanceof ArchiveInput.ZipFileSystemInput) {
                return new ZipFileSystemOutput(outputFile.toPath());
            }
            return new ZipFileOutput(outputFile);
        }
        if (input instanceof ArchiveInput.TarFileInput) {
            return new TarFileOutput(outputFile);
        }
        return new ZipFileOutput(outputFile);
    }

    public static class DirectoryOutput
    extends AbstractFileSystemOutput
    implements FileBacked {
        public DirectoryOutput(Path path) throws IOException {
            super(DirectoryOutput.checkCreate(path));
        }

        private static Path checkCreate(Path path) throws IOException {
            if (!Files.exists(path, new LinkOption[0])) {
                Files.createDirectory(path, new FileAttribute[0]);
            } else {
                Preconditions.checkState((boolean)Files.isDirectory(path, new LinkOption[0]), (String)"Path already exists and is not a directory: %s", (Object)path);
            }
            return path.toAbsolutePath();
        }

        @Override
        public String getName() {
            return this.upstreamPath.toString();
        }

        @Override
        public File getInProgressFile() {
            return this.upstreamPath.toFile();
        }

        @Override
        public File getDestinationFile() {
            return this.upstreamPath.toFile();
        }

        @Override
        public ArchiveInput getCompletedInput() throws IOException {
            return new ArchiveInput.DirectoryInput(this.upstreamPath);
        }
    }

    public static class TarFileOutput
    extends AbstractTarOutput
    implements FileBacked {
        private File outputFile;
        private File inProgressFile;

        public TarFileOutput(File outputFile) throws IOException {
            this(outputFile, new File(outputFile.getAbsolutePath() + ".tmp"));
        }

        public TarFileOutput(File outputFile, File inProgressFile) throws IOException {
            super(new TarArchiveOutputStream((OutputStream)new BufferedOutputStream(new FileOutputStream(inProgressFile))));
            this.outputFile = outputFile;
            this.inProgressFile = inProgressFile;
        }

        @Override
        public String getName() {
            return this.outputFile.getAbsolutePath();
        }

        @Override
        public File getInProgressFile() {
            return this.inProgressFile;
        }

        @Override
        public File getDestinationFile() {
            return this.outputFile;
        }

        @Override
        public void close() throws IOException {
            if (this.tout == null) {
                return;
            }
            super.close();
            if (!this.inProgressFile.equals(this.outputFile)) {
                Files.move(this.inProgressFile.toPath(), this.outputFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
        }

        @Override
        public ArchiveInput getCompletedInput() throws IOException {
            Preconditions.checkState((this.tout == null ? 1 : 0) != 0, (Object)"ZipOutputStream is still open");
            return new ArchiveInput.TarFileInput(this.outputFile);
        }
    }

    public static class ApacheZipFileOutput
    extends AbstractApacheZipOutput
    implements FileBacked {
        private File outputFile;
        private File inProgressFile;

        public ApacheZipFileOutput(File outputFile) throws IOException {
            this(outputFile, ArchiveOutput.getDefaultInProgressFile(outputFile));
        }

        public ApacheZipFileOutput(File outputFile, File inProgressFile) throws IOException {
            super(new ZipArchiveOutputStream(inProgressFile), true);
            this.outputFile = outputFile;
            this.inProgressFile = inProgressFile;
        }

        @Override
        public void close() throws IOException {
            if (this.zout == null) {
                return;
            }
            super.close();
            if (!this.inProgressFile.equals(this.outputFile)) {
                Files.move(this.inProgressFile.toPath(), this.outputFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
        }

        @Override
        public File getInProgressFile() {
            return this.inProgressFile;
        }

        @Override
        public File getDestinationFile() {
            return this.outputFile;
        }

        @Override
        public ArchiveInput getCompletedInput() throws IOException {
            Preconditions.checkState((this.zout == null ? 1 : 0) != 0, (Object)"ZipOutputStream is still open");
            return new ArchiveInput.ApacheZipFileInput(this.outputFile);
        }
    }

    public static class ZipFileSystemOutput
    extends AbstractFileSystemOutput
    implements FileBacked {
        private Path inProgressPath;
        private Path destinationPath;

        public ZipFileSystemOutput(Path path) throws IOException {
            this(path, Path.of(path.toString() + ".tmp", new String[0]));
        }

        public ZipFileSystemOutput(Path destinationPath, Path inProgressPath) throws IOException {
            super(ZipFileSystemOutput.initFS(inProgressPath));
            this.destinationPath = destinationPath.toAbsolutePath();
            this.inProgressPath = inProgressPath.toAbsolutePath();
        }

        private static FileSystem initFS(Path path) throws IOException {
            URI uri = URI.create("jar:" + path.toUri().toString());
            Map<String, String> env = Map.of("create", "true");
            return FileSystems.newFileSystem(uri, env);
        }

        @Override
        public String getName() {
            return this.destinationPath.toString();
        }

        @Override
        public File getInProgressFile() {
            return this.inProgressPath.toFile();
        }

        @Override
        public File getDestinationFile() {
            return this.destinationPath.toFile();
        }

        @Override
        public ArchiveInput getCompletedInput() throws IOException {
            Preconditions.checkState((this.fs == null ? 1 : 0) != 0, (Object)"Output FileSystem is still open");
            return new ArchiveInput.ZipFileSystemInput(this.destinationPath);
        }

        @Override
        public void close() throws IOException {
            super.close();
            if (!this.destinationPath.equals(this.inProgressPath)) {
                Files.move(this.inProgressPath, this.destinationPath, StandardCopyOption.REPLACE_EXISTING);
            }
        }
    }

    public static class ZipFileOutput
    implements FileBacked {
        private File inProgressFile;
        private File outputFile;
        private ZipOutputStream zout;

        public ZipFileOutput(File outputFile) throws IOException {
            this(outputFile, ArchiveOutput.getDefaultInProgressFile(outputFile));
        }

        public ZipFileOutput(File outputFile, File inProgressFile) throws IOException {
            this.outputFile = outputFile;
            this.inProgressFile = inProgressFile;
            this.zout = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(inProgressFile)));
        }

        public ZipFileOutput(ZipOutputStream zout) {
            this.inProgressFile = this.outputFile = new File(zout.toString());
        }

        @Override
        public void putNextEntry(String name) throws IOException {
            this.zout.putNextEntry(new ZipEntry(name));
        }

        @Override
        public OutputStream getOutputStream() throws IOException {
            return this.zout;
        }

        @Override
        public void closeEntry() throws IOException {
            this.zout.flush();
            this.zout.closeEntry();
        }

        @Override
        public void close() throws IOException {
            if (this.zout == null) {
                return;
            }
            this.zout.close();
            this.zout = null;
            if (!this.inProgressFile.equals(this.outputFile)) {
                Files.move(this.inProgressFile.toPath(), this.outputFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
        }

        @Override
        public File getInProgressFile() {
            return this.inProgressFile;
        }

        @Override
        public File getDestinationFile() {
            return this.outputFile;
        }

        @Override
        public ArchiveInput getCompletedInput() throws IOException {
            Preconditions.checkState((this.zout == null ? 1 : 0) != 0, (Object)"ZipOutputStream is still open");
            return new ArchiveInput.ZipFileInput(this.outputFile);
        }
    }

    public static abstract class AbstractFileSystemOutput
    implements ArchiveOutput {
        protected FileSystem fs;
        protected Path upstreamPath;
        private Path currentPath;
        private OutputStream currentOutputStream;

        public AbstractFileSystemOutput(FileSystem fs) {
            this.fs = fs;
        }

        public AbstractFileSystemOutput(Path upstreamPath) {
            this.upstreamPath = upstreamPath;
        }

        @Override
        public void putNextEntry(String name) throws IOException {
            Preconditions.checkState((this.currentPath == null ? 1 : 0) != 0, (String)"Never closed previous entry (%s)", (Object)this.currentPath);
            this.currentPath = this.upstreamPath == null ? this.fs.getPath(name, new String[0]) : this.upstreamPath.resolve(name);
            Path parent = this.currentPath.getParent();
            if (parent != null) {
                Files.createDirectories(parent, new FileAttribute[0]);
            }
            if (name.endsWith(ArchiveOutput.SEPERATOR)) {
                if (Files.exists(this.currentPath, new LinkOption[0])) {
                    Preconditions.checkState((boolean)Files.isDirectory(this.currentPath, new LinkOption[0]), (String)"Cannot create directory '%s' because path already exists and isn't a directory: %s", (Object)name, (Object)this.currentPath);
                } else {
                    Files.createDirectory(this.currentPath, new FileAttribute[0]);
                }
            }
        }

        @Override
        public OutputStream getOutputStream() throws IOException {
            Preconditions.checkNotNull((Object)this.currentPath, (Object)"Trying to getOutputStream without first calling initNewEntry");
            this.currentOutputStream = new BufferedOutputStream(Files.newOutputStream(this.currentPath, this.fs == null ? StandardOpenOption.CREATE : StandardOpenOption.CREATE_NEW));
            return this.currentOutputStream;
        }

        @Override
        public void closeEntry() throws IOException {
            Preconditions.checkNotNull((Object)this.currentPath, (Object)"Trying to closeEntry without first calling initNewEntry");
            if (this.currentOutputStream == null) {
                Preconditions.checkState((boolean)Files.isDirectory(this.currentPath, new LinkOption[0]), (String)"Called closeEntry() but no output streams currently open and not a directory; currentPath=%s", (Object)this.currentPath);
            } else {
                this.currentOutputStream.close();
            }
            this.currentPath = null;
            this.currentOutputStream = null;
        }

        @Override
        public void close() throws IOException {
            if (this.fs == null) {
                return;
            }
            this.fs.close();
            this.fs = null;
        }
    }

    public static abstract class AbstractTarOutput
    implements ArchiveOutput {
        protected TarArchiveOutputStream tout;
        private String currentEntry = null;
        private ByteArrayOutputStream currentOutput = null;

        public AbstractTarOutput(TarArchiveOutputStream tout) {
            this.tout = tout;
        }

        @Override
        public void putNextEntry(String name) throws IOException {
            Preconditions.checkState((this.currentEntry == null ? 1 : 0) != 0, (String)"Called putNextEntry(%s) with an already open entry (%s)", (Object)name, (Object)this.currentEntry);
            this.currentEntry = name;
        }

        @Override
        public OutputStream getOutputStream() throws IOException {
            Preconditions.checkNotNull((Object)this.currentEntry, (Object)"Must putNextEntry before calling getOutputStream");
            Preconditions.checkState((this.currentOutput == null ? 1 : 0) != 0, (Object)"Can't call getOutputStream multiple times, must closeEntry and putNextEntry");
            this.currentOutput = new ByteArrayOutputStream(0x500000);
            return this.currentOutput;
        }

        @Override
        public void closeEntry() throws IOException {
            Preconditions.checkNotNull((Object)this.currentEntry, (Object)"Can't closeEntry because it was never opened");
            TarArchiveEntry entry = new TarArchiveEntry(this.currentEntry);
            if (this.currentOutput == null) {
                Preconditions.checkState((boolean)entry.isDirectory(), (String)"Output stream never written for %s, but it's not a directory (doesn't end with '/')", (Object)this.currentEntry);
                this.tout.putArchiveEntry(entry);
            } else {
                entry.setSize((long)this.currentOutput.size());
                this.tout.putArchiveEntry(entry);
                this.currentOutput.writeTo((OutputStream)this.tout);
                this.tout.closeArchiveEntry();
            }
            this.currentOutput = null;
            this.currentEntry = null;
        }

        @Override
        public void close() throws IOException {
            if (this.tout == null) {
                return;
            }
            this.tout.close();
        }
    }

    public static class ParallelZipFileOutput
    extends ApacheZipFileOutput {
        private static final boolean D = false;
        private ArrayDeque<CompletableFuture<AsynchronousApacheZipper>> zipFutures;
        private ArrayDeque<AsynchronousApacheZipper> zippers;
        private CompletableFuture<Void> writeFuture;
        private boolean preserveOrder;
        private int maxThreads;
        private int threadsInitialized;
        private CopyAvoidantInMemorySeekableByteChannel secondaryZipBuffer;
        private boolean trackBlockingTimes;
        private Stopwatch overallWatch = null;
        private Stopwatch blockingWriteWatch = null;
        private Stopwatch blockingZipWatch = null;
        private Runnable postProcessRun;
        private ExecutorService postProcessExec = ExecutorUtils.singleTaskRejectingExecutor();
        private AsynchronousApacheZipper currentZipper;
        private static final DecimalFormat pDF = new DecimalFormat("0.#%");

        public ParallelZipFileOutput(File outputFile, int threads) throws IOException {
            this(outputFile, threads, true);
        }

        public ParallelZipFileOutput(File outputFile, int threads, boolean preserveOrder) throws IOException {
            super(outputFile);
            this.init(threads, preserveOrder);
        }

        public ParallelZipFileOutput(File outputFile, File inProgressFile, int threads, boolean preserveOrder) throws IOException {
            super(outputFile, inProgressFile);
            this.init(threads, preserveOrder);
        }

        private void init(int threads, boolean preserveOrder) {
            this.preserveOrder = preserveOrder;
            Preconditions.checkState((threads > 0 ? 1 : 0) != 0, (Object)"Must supply at least 1 thread");
            this.maxThreads = threads;
            this.threadsInitialized = 0;
            this.zipFutures = new ArrayDeque(threads);
            this.zippers = new ArrayDeque(threads);
            this.postProcessRun = new Runnable(){

                @Override
                public void run() {
                    try {
                        this.processZipFutures();
                    }
                    catch (IOException e) {
                        throw ExceptionUtils.asRuntimeException((Throwable)e);
                    }
                }
            };
        }

        public void setTrackBlockingTimes(boolean track) {
            if (this.trackBlockingTimes) {
                this.overallWatch.stop();
                this.blockingWriteWatch.stop();
                this.blockingWriteWatch.stop();
                this.blockingZipWatch.stop();
            }
            if (track) {
                this.trackBlockingTimes = true;
                this.overallWatch = Stopwatch.createStarted();
                this.blockingWriteWatch = Stopwatch.createUnstarted();
                this.blockingZipWatch = Stopwatch.createUnstarted();
            } else {
                this.trackBlockingTimes = false;
                this.overallWatch = null;
                this.blockingWriteWatch = null;
                this.blockingZipWatch = null;
            }
        }

        public String getBlockingTimeStats() {
            Preconditions.checkState((boolean)this.trackBlockingTimes);
            double totSecs = (double)this.overallWatch.elapsed(TimeUnit.SECONDS) / 1000.0;
            double writeSecs = (double)this.blockingWriteWatch.elapsed(TimeUnit.SECONDS) / 1000.0;
            double zipSecs = (double)this.blockingZipWatch.elapsed(TimeUnit.SECONDS) / 1000.0;
            return pDF.format(writeSecs / totSecs) + " blocking writes, " + pDF.format(zipSecs / totSecs) + " blocking zips, " + this.threadsInitialized + ArchiveOutput.SEPERATOR + this.maxThreads + " workers initialized";
        }

        private synchronized void joinAllWriters() throws IOException {
            while (!this.zipFutures.isEmpty()) {
                AsynchronousApacheZipper zip = this.zipFutures.remove().join();
                this.transferZippedData(zip);
            }
            if (this.writeFuture != null) {
                if (this.trackBlockingTimes) {
                    this.blockingWriteWatch.start();
                }
                this.writeFuture.join();
                if (this.trackBlockingTimes) {
                    this.blockingWriteWatch.stop();
                }
                this.writeFuture = null;
            }
        }

        private synchronized void processZipFutures() throws IOException {
            if (this.writeFuture != null && !this.writeFuture.isDone()) {
                return;
            }
            for (CompletableFuture<AsynchronousApacheZipper> peek : this.zipFutures) {
                if (this.writeFuture != null && !this.writeFuture.isDone()) break;
                if (peek.isDone()) {
                    Preconditions.checkState((boolean)this.zipFutures.remove(peek));
                    AsynchronousApacheZipper zip = peek.join();
                    this.transferZippedData(zip);
                    break;
                }
                if (!this.preserveOrder) continue;
                break;
            }
        }

        private synchronized void transferZippedData(AsynchronousApacheZipper zipper) throws IOException {
            if (this.writeFuture != null) {
                if (this.trackBlockingTimes) {
                    this.blockingWriteWatch.start();
                }
                this.writeFuture.join();
                if (this.trackBlockingTimes) {
                    this.blockingWriteWatch.stop();
                }
            }
            Preconditions.checkState((boolean)zipper.hasEntry());
            if (!zipper.hasEntryData()) {
                super.putNextEntry(zipper.getEntryDestName());
                super.closeEntry();
                this.writeFuture = CompletableFuture.completedFuture(null);
                this.zippers.add(zipper);
                this.processZipFutures();
            } else {
                this.writeFuture = zipper.rawTransferFuture(this);
                this.secondaryZipBuffer = zipper.swapCompressedDataBuffer(this.secondaryZipBuffer);
                this.zippers.add(zipper);
                this.writeFuture.thenRunAsync(this.postProcessRun, this.postProcessExec);
            }
        }

        @Override
        public synchronized void close() throws IOException {
            this.joinAllWriters();
            this.postProcessExec.shutdown();
            Preconditions.checkState((boolean)this.zipFutures.isEmpty());
            Preconditions.checkState((this.writeFuture == null ? 1 : 0) != 0);
            this.zipFutures = null;
            this.zippers = null;
            if (this.trackBlockingTimes) {
                if (this.overallWatch.isRunning()) {
                    this.overallWatch.stop();
                }
                if (this.blockingWriteWatch.isRunning()) {
                    this.blockingWriteWatch.stop();
                }
                if (this.blockingZipWatch.isRunning()) {
                    this.blockingZipWatch.stop();
                }
            }
            super.close();
        }

        private void prepareZipper() throws IOException {
            this.processZipFutures();
            Preconditions.checkState((this.currentZipper == null ? 1 : 0) != 0);
            this.currentZipper = this.zippers.poll();
            if (this.currentZipper == null) {
                if (this.threadsInitialized < this.maxThreads) {
                    this.currentZipper = new AsynchronousApacheZipper();
                    ++this.threadsInitialized;
                } else {
                    Preconditions.checkState((!this.zipFutures.isEmpty() ? 1 : 0) != 0);
                    CompletableFuture<AsynchronousApacheZipper> future = this.zipFutures.remove();
                    if (this.trackBlockingTimes) {
                        this.blockingZipWatch.start();
                    }
                    AsynchronousApacheZipper zip = future.join();
                    if (this.trackBlockingTimes) {
                        this.blockingZipWatch.stop();
                    }
                    this.transferZippedData(zip);
                    Preconditions.checkNotNull(this.writeFuture);
                    this.currentZipper = this.zippers.remove();
                }
            }
            this.currentZipper.reset();
        }

        @Override
        public synchronized void putNextEntry(String name) throws IOException {
            this.prepareZipper();
            this.currentZipper.putNextEntry(name);
        }

        @Override
        public synchronized OutputStream getOutputStream() throws IOException {
            Preconditions.checkNotNull((Object)this.currentZipper);
            return this.currentZipper.getOutputStream();
        }

        @Override
        public synchronized void closeEntry() throws IOException {
            Preconditions.checkNotNull((Object)this.currentZipper);
            CompletableFuture<AsynchronousApacheZipper> future = this.currentZipper.closeEntry();
            future.thenRunAsync(this.postProcessRun, this.postProcessExec);
            this.currentZipper = null;
            this.zipFutures.add(future);
        }

        @Override
        public synchronized void transferFrom(ArchiveInput input, String sourceName, String destName) throws IOException {
            this.prepareZipper();
            CompletableFuture<AsynchronousApacheZipper> future = this.currentZipper.transferFrom(input, sourceName, destName);
            future.thenRunAsync(this.postProcessRun, this.postProcessExec);
            this.currentZipper = null;
            this.zipFutures.add(future);
        }
    }

    public static class AsynchronousZipFileOutput
    extends ApacheZipFileOutput {
        private CopyAvoidantInMemorySeekableByteChannel zippingBuffer;
        private CopyAvoidantInMemorySeekableByteChannel writingBuffer;
        private String currentEntry;
        private InMemoryZipOutput currentOutput;
        private AsynchronousApacheZipper.WriteShieldAfterCloseOutputStream currentOutputStream;
        private CompletableFuture<?> writeFuture;

        public AsynchronousZipFileOutput(File outputFile) throws IOException {
            super(outputFile);
            this.init();
        }

        public AsynchronousZipFileOutput(File outputFile, File inProgressFile) throws IOException {
            super(outputFile, inProgressFile);
            this.init();
        }

        private void init() {
            this.zippingBuffer = new CopyAvoidantInMemorySeekableByteChannel(0x500000);
            this.zippingBuffer.setCloseable(false);
            this.writingBuffer = new CopyAvoidantInMemorySeekableByteChannel(0x500000);
            this.writingBuffer.setCloseable(false);
        }

        private void startAsyncWrite() throws IOException {
            if (this.writeFuture != null) {
                this.writeFuture.join();
            }
            Preconditions.checkNotNull((Object)this.currentEntry);
            final String entryName = this.currentEntry;
            if (this.currentOutput == null) {
                super.putNextEntry(entryName);
                super.closeEntry();
                this.writeFuture = CompletableFuture.completedFuture(null);
            } else {
                this.currentOutput.close();
                this.zippingBuffer.position(0L);
                final ArchiveInput.InMemoryZipInput input = new ArchiveInput.InMemoryZipInput(this.zippingBuffer);
                this.writeFuture = CompletableFuture.runAsync(new Runnable(){
                    final /* synthetic */ AsynchronousZipFileOutput this$0;
                    {
                        this.this$0 = this$0;
                    }

                    @Override
                    public void run() {
                        try {
                            this.this$0.rawTransferApache(input, entryName, entryName);
                        }
                        catch (IOException e) {
                            throw ExceptionUtils.asRuntimeException((Throwable)e);
                        }
                    }
                });
            }
            this.currentEntry = null;
            this.currentOutput = null;
            CopyAvoidantInMemorySeekableByteChannel tmpBuffer = this.writingBuffer;
            this.writingBuffer = this.zippingBuffer;
            this.zippingBuffer = tmpBuffer;
        }

        @Override
        public synchronized void putNextEntry(String name) throws IOException {
            Preconditions.checkState((this.currentEntry == null ? 1 : 0) != 0);
            Preconditions.checkState((this.currentOutput == null ? 1 : 0) != 0);
            this.zippingBuffer.truncate(0L);
            this.currentEntry = name;
        }

        @Override
        public synchronized OutputStream getOutputStream() throws IOException {
            Preconditions.checkNotNull((Object)this.currentEntry, (Object)"Called getOutputStream() without first calling putNextEntry()");
            Preconditions.checkState((this.currentOutput == null ? 1 : 0) != 0, (Object)"Can't call getOutputStream() twice on the same entry");
            this.currentOutput = new InMemoryZipOutput(true, this.zippingBuffer);
            this.currentOutput.putNextEntry(this.currentEntry);
            this.currentOutputStream = new AsynchronousApacheZipper.WriteShieldAfterCloseOutputStream(this.currentOutput.getOutputStream());
            return this.currentOutputStream;
        }

        @Override
        public synchronized void closeEntry() throws IOException {
            Preconditions.checkNotNull((Object)this.currentEntry, (Object)"Called closeEntry() without first calling putNextEntry()");
            if (this.currentOutput != null) {
                this.currentOutput.closeEntry();
                this.currentOutputStream.lock();
                this.currentOutputStream = null;
            }
            this.startAsyncWrite();
        }

        @Override
        public synchronized void transferFrom(ArchiveInput input, String sourceName, String destName) throws IOException {
            Preconditions.checkState((this.currentEntry == null && this.currentOutput == null ? 1 : 0) != 0, (Object)"Called transferFrom(..) when another entry was open?");
            this.putNextEntry(destName);
            this.currentOutput = new InMemoryZipOutput(true, this.zippingBuffer);
            this.currentOutput.transferFrom(input, sourceName, destName);
            this.startAsyncWrite();
        }

        @Override
        public void close() throws IOException {
            if (this.writeFuture != null) {
                this.writeFuture.join();
            }
            this.writeFuture = null;
            this.zippingBuffer = null;
            this.writingBuffer = null;
            super.close();
        }
    }

    public static class InMemoryZipOutput
    extends AbstractApacheZipOutput {
        private boolean compressed;
        private CopyAvoidantInMemorySeekableByteChannel byteChannel;

        public InMemoryZipOutput(boolean compressed) {
            this(compressed, 0x3200000);
        }

        public InMemoryZipOutput(boolean compressed, int initialSize) {
            this(compressed, new CopyAvoidantInMemorySeekableByteChannel(initialSize));
        }

        public InMemoryZipOutput(boolean compressed, CopyAvoidantInMemorySeekableByteChannel byteChannel) {
            super(new ZipArchiveOutputStream((SeekableByteChannel)byteChannel), compressed);
            this.compressed = compressed;
            this.byteChannel = byteChannel;
        }

        @Override
        public String getName() {
            return "In Memory (" + (this.compressed ? "zipped" : "uncompressed") + ")";
        }

        @Override
        public void putNextEntry(String name) throws IOException {
            super.putNextEntry(name);
        }

        @Override
        public void closeEntry() throws IOException {
            super.closeEntry();
        }

        @Override
        public ArchiveInput.InMemoryZipInput getCompletedInput() throws IOException {
            Preconditions.checkState((this.zout == null ? 1 : 0) != 0, (Object)"Not closed?");
            CopyAvoidantInMemorySeekableByteChannel copy = this.byteChannel.copy();
            copy.position(0L);
            return new ArchiveInput.InMemoryZipInput(copy);
        }
    }

    public static abstract class AbstractApacheZipOutput
    implements ArchiveOutput {
        protected ZipArchiveOutputStream zout;

        public AbstractApacheZipOutput(ZipArchiveOutputStream zout, boolean compressed) {
            this.zout = zout;
            if (!compressed) {
                zout.setMethod(0);
            }
        }

        @Override
        public void putNextEntry(String name) throws IOException {
            this.zout.putArchiveEntry(new ZipArchiveEntry(name));
        }

        @Override
        public OutputStream getOutputStream() throws IOException {
            return this.zout;
        }

        @Override
        public void closeEntry() throws IOException {
            this.zout.flush();
            this.zout.closeArchiveEntry();
        }

        @Override
        public void transferFrom(ArchiveInput input, String sourceName, String destName) throws IOException {
            if (input instanceof ArchiveInput.AbstractApacheZipInput) {
                ArchiveInput.AbstractApacheZipInput apache = (ArchiveInput.AbstractApacheZipInput)input;
                this.rawTransferApache(apache, sourceName, destName);
            } else {
                ArchiveOutput.super.transferFrom(input, sourceName, destName);
            }
        }

        protected void rawTransferApache(ArchiveInput.AbstractApacheZipInput apache, String sourceName, String destName) throws IOException {
            ZipArchiveEntry sourceEntry = apache.getEntry(sourceName);
            ZipArchiveEntry outEntry = new ZipArchiveEntry(destName);
            outEntry.setCompressedSize(sourceEntry.getCompressedSize());
            outEntry.setCrc(sourceEntry.getCrc());
            outEntry.setExternalAttributes(sourceEntry.getExternalAttributes());
            outEntry.setExtra(sourceEntry.getExtra());
            outEntry.setExtraFields(sourceEntry.getExtraFields());
            outEntry.setGeneralPurposeBit(sourceEntry.getGeneralPurposeBit());
            outEntry.setInternalAttributes(sourceEntry.getInternalAttributes());
            outEntry.setMethod(sourceEntry.getMethod());
            outEntry.setRawFlag(sourceEntry.getRawFlag());
            outEntry.setSize(sourceEntry.getSize());
            this.zout.addRawArchiveEntry(outEntry, apache.getRawInputStream(sourceEntry));
            this.zout.flush();
        }

        @Override
        public void close() throws IOException {
            if (this.zout == null) {
                return;
            }
            this.zout.close();
            this.zout = null;
        }
    }

    public static interface FileBacked
    extends ArchiveOutput {
        public File getInProgressFile();

        public File getDestinationFile();

        @Override
        default public String getName() {
            File destFile = this.getDestinationFile();
            if (destFile == null) {
                return null;
            }
            return destFile.getAbsolutePath();
        }
    }
}

