package org.opensha.commons.calc;

import java.util.ArrayList;
import java.util.List;

import org.opensha.commons.data.function.AbstractXY_DataSet;
import org.opensha.commons.data.function.ArbDiscrEmpiricalDistFunc;
import org.opensha.commons.data.function.ArbitrarilyDiscretizedFunc;
import org.opensha.commons.data.function.XY_DataSetList;
/**
 * <p>Title:  FractileCurveCalculator</p>
 * <p>Description: This class calculates fractiles from a list of discretized functions (e.g., hazzard curves) and their relative weights</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: </p>
 * @author unascribed
 * @version 1.0
 */

public class FractileCurveCalculator {

	// function list to save the curves
	private XY_DataSetList funcList;
	// save the relative weight of each curve
	private List<Double> relativeWeights;
	// save the number of X values
	private int num;
	// vector to save the empirical distributions
	private List<ArbDiscrEmpiricalDistFunc> empiricalDists;

	// Error Strings to be dispalyed
	private final static String ERROR_WEIGHTS =
			"Error! Number of weights should be equal to number of curves";
	private final static String ERROR_LIST = "No curves exist in the list";
	private final static String ERROR_POINTS =
			"Number of points in each curve should be same";


	/**
	 * Constructor : Calls the set function
	 * @param functionList : List of curves for which fractile needs to be calculated
	 * @param relativeWts : weight assigned to each curves. It expects the ArrayList
	 *  to contain Double values
	 */
	public FractileCurveCalculator(XY_DataSetList functionList,
			List<Double> relativeWts) {
		set(functionList, relativeWts);
	}



	/**
	 * It accepts the function list for curves and relative weight
	 * assigned to each curve.
	 * It checks for following condition :
	 *   1. Number of weights = number of curves in  list
	 *          (i.e. functionList.size() = relativeWts.size()  )
	 *   2. Number of X values in all curves are same
	 *
	 * It makes following asssumption:
	 *   X values for in the curves are same
	 *
	 * @param functionList : List of curves for which fractile needs to be calculated
	 * @param relativeWts : Weight assigned to each curve, stored as a Double objects in this
	 * array list in the same order as in the functionList. Note that these values do not have to
	 * be normalized (they don't have to sum to 1.0) - this normalization is taken care of internally.
	 */
	public void set(XY_DataSetList functionList,
			List<Double> relativeWts) {

		// check that number of weights are equal to number of curves give
		if(functionList.size()!=relativeWts.size()) throw new RuntimeException(ERROR_WEIGHTS);

		// check that curve list is not empty
		int numFunctions = functionList.size();
		if(numFunctions==0) throw new RuntimeException(ERROR_LIST);

		// check  that all curves in list have same number of X values
		int numPoints = functionList.get(0).size();
		for(int i=1; i<numFunctions; ++i)
			if(functionList.get(i).size()!=numPoints) throw new RuntimeException(ERROR_POINTS);

		this.funcList = functionList;
		this.relativeWeights = relativeWts; // these do not need to be normalized
		this.num = numPoints;

		//ArrayList for saving empirical distributions
		empiricalDists = new ArrayList<ArbDiscrEmpiricalDistFunc>();

		// make a empirical dist for each X value
		for(int i=0; i<num; ++i) {
			ArbDiscrEmpiricalDistFunc empirical = new ArbDiscrEmpiricalDistFunc();
			for(int j=0; j<numFunctions; ++j)
				empirical.set(funcList.get(j).getY(i),
						((Double)relativeWeights.get(j)).doubleValue());
			empiricalDists.add(empirical);
			//      System.out.println("111  i="+i+"; dist="+empirical.toString());
		}

	}

	/**
	 * This computes the mean curve from the list of functions (and their associated
	 * weights).
	 * 
	 * TODO this should use empiricalDists
	 * @return
	 */
	public AbstractXY_DataSet getMeanCurve() {
		AbstractXY_DataSet result = (AbstractXY_DataSet) funcList.get(0).deepClone();
		double wt, totWt=0;
		int numPoints = funcList.get(0).size();
		int numFuncs = funcList.size();
		int i, f;

		// initialize function to zero
		for(i=0;i<numPoints;i++)
			result.set(i,0.0);

		// add all functions (weighted) together
		for(f=0;f<numFuncs;f++) {
			wt = ((Double)relativeWeights.get(f)).doubleValue();
			totWt += wt;
			for(i=0;i<numPoints;i++)
				result.set(i, result.getY(i) + wt*funcList.get(f).getY(i) );
		}

		// now normalize by the total weight
		// initialize result to zero
		for(i=0;i<numPoints;i++)
			result.set(i,result.getY(i)/totWt);

		result.setName("Mean");

		return result;
	}

	/**
	 * This returns a curve with the minimum value among all the curves.
	 * 
	 * TODO this should use empiricalDists
	 * 
	 * @return
	 */
	public AbstractXY_DataSet getMinimumCurve() {
		AbstractXY_DataSet result = (AbstractXY_DataSet) funcList.get(0).deepClone();

		// initialize function to large values
		for(int i=0;i<result.size();i++)
			result.set(i,Double.MAX_VALUE);

		// loop over functions & x-axis values
		for(int f=0;f<funcList.size();f++) {
			for(int i=0;i<result.size();i++) {
				if(funcList.get(f).getY(i)<result.getY(i))
					result.set(i, funcList.get(f).getY(i) );
			}
		}
		result.setName("Minimum");
		return result;
	}


	/**
	 * This returns a curve with the maximum value among all the curves.
	 * 
	 * TODO this should use empiricalDists
	 * 
	 * @return
	 */
	public AbstractXY_DataSet getMaximumCurve() {
		AbstractXY_DataSet result = (AbstractXY_DataSet) funcList.get(0).deepClone();

		// initialize function to large values
		for(int i=0;i<result.size();i++)
			result.set(i,-Double.MAX_VALUE);

		// loop over functions & x-axis values
		for(int f=0;f<funcList.size();f++) {
			for(int i=0;i<result.size();i++) {
				if(funcList.get(f).getY(i)>result.getY(i))
					result.set(i, funcList.get(f).getY(i) );
			}
		}
		result.setName("Maximum");
		return result;
	}



	/**
	 *  This returns the fractile curve corresponding to the specified fraction
	 * @param fraction
	 * @return
	 */
	public ArbitrarilyDiscretizedFunc getStdDev() {
		// function for the result
		ArbitrarilyDiscretizedFunc result = new ArbitrarilyDiscretizedFunc();
		for(int i=0; i<num; ++i) {
			result.set(funcList.get(0).getX(i),
					((ArbDiscrEmpiricalDistFunc)empiricalDists.get(i)).getStdDev());
		}
		result.setName("stdDev");
		return result;
	}



	/**
	 *  This returns the fractile curve corresponding to the specified fraction
	 * @param fraction
	 * @return
	 */
	public AbstractXY_DataSet getFractile(double fraction) {
		AbstractXY_DataSet result = (AbstractXY_DataSet) funcList.get(0).deepClone();
		// function to save the result
		//    ArbitrarilyDiscretizedFunc result = new ArbitrarilyDiscretizedFunc();
		for(int i=0; i<num; ++i) {
			result.set(funcList.get(0).getX(i),
					((ArbDiscrEmpiricalDistFunc)empiricalDists.get(i)).getDiscreteFractile(fraction));
		}
		result.setName(fraction+" fractile");
		return result;
	}
	
	public ArbDiscrEmpiricalDistFunc getEmpiricalDist(int xIndex) {
		return empiricalDists.get(xIndex);
	}
}
