/*
 * Decompiled with CFR 0.152.
 */
package org.opensha.commons.geo;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.primitives.Doubles;
import com.google.gson.TypeAdapter;
import com.google.gson.annotations.JsonAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;
import java.awt.geom.Area;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.dom4j.Attribute;
import org.dom4j.Element;
import org.opensha.commons.data.Named;
import org.opensha.commons.geo.BorderType;
import org.opensha.commons.geo.GeoTools;
import org.opensha.commons.geo.GriddedRegion;
import org.opensha.commons.geo.Location;
import org.opensha.commons.geo.LocationList;
import org.opensha.commons.geo.LocationUtils;
import org.opensha.commons.geo.LocationVector;
import org.opensha.commons.geo.json.Feature;
import org.opensha.commons.geo.json.FeatureProperties;
import org.opensha.commons.geo.json.Geometry;
import org.opensha.commons.metadata.XMLSaveable;

@JsonAdapter(value=Adapter.class)
public class Region
implements Serializable,
XMLSaveable,
Named {
    private static final long serialVersionUID = 1L;
    private LocationList border;
    private ArrayList<LocationList> interiors;
    Area area;
    private static final double WEDGE_WIDTH = 10.0;
    private static final double GC_SEGMENT = 100.0;
    public static final String XML_METADATA_NAME = "Region";
    public static final String XML_METADATA_OUTLINE_NAME = "OutlineLocations";
    protected static final String NAME_DEFAULT = "Unnamed Region";
    private String name = "Unnamed Region";
    private static final boolean SUBTRACT_DEBUG = false;
    private BorderType type;
    private static final int max_split_recusrsion = 20;

    private Region() {
    }

    public Region(Location loc1, Location loc2) {
        Preconditions.checkNotNull((Object)loc1, (Object)"Supplied location (1) is null");
        Preconditions.checkNotNull((Object)loc1, (Object)"Supplied location (2) is null");
        double lat1 = loc1.getLatitude();
        double lat2 = loc2.getLatitude();
        double lon1 = loc1.getLongitude();
        double lon2 = loc2.getLongitude();
        Preconditions.checkArgument((lat1 != lat2 ? 1 : 0) != 0, (Object)"Input lats cannot be the same");
        Preconditions.checkArgument((lon1 != lon2 ? 1 : 0) != 0, (Object)"Input lons cannot be the same");
        LocationList ll = new LocationList();
        double minLat = Math.min(lat1, lat2);
        double minLon = Math.min(lon1, lon2);
        double maxLat = Math.max(lat1, lat2);
        double maxLon = Math.max(lon1, lon2);
        double offset = 1.0E-12;
        maxLat += maxLat <= 90.0 - offset ? offset : 0.0;
        double d = maxLon <= 180.0 - offset ? offset : 0.0;
        ll.add(new Location(minLat -= minLat >= -90.0 + offset ? offset : 0.0, minLon -= minLon >= -180.0 + offset ? offset : 0.0));
        ll.add(new Location(minLat, maxLon += d));
        ll.add(new Location(maxLat, maxLon));
        ll.add(new Location(maxLat, minLon));
        this.initBorderedRegion(ll, BorderType.MERCATOR_LINEAR);
    }

    public Region(LocationList border, BorderType type) {
        Preconditions.checkNotNull((Object)border, (Object)"Supplied border is null");
        Preconditions.checkArgument((border.size() >= 3 ? 1 : 0) != 0, (Object)"Supplied border must have at least 3 vertices");
        if (type == null) {
            type = BorderType.MERCATOR_LINEAR;
        }
        this.initBorderedRegion(border, type);
    }

    public Region(Location center, double radius) {
        Preconditions.checkNotNull((Object)center, (Object)"Supplied center Location is null");
        Preconditions.checkArgument((radius > 0.0 && radius <= 1000.0 ? 1 : 0) != 0, (String)"Radius [%s] is out of [0 1000] km range", (Object)radius);
        this.initCircularRegion(center, radius);
    }

    public Region(LocationList line, double buffer) {
        Preconditions.checkNotNull((Object)line, (Object)"Supplied LocationList is null");
        Preconditions.checkArgument((!line.isEmpty() ? 1 : 0) != 0, (Object)"Supplied LocationList is empty");
        Preconditions.checkArgument((buffer > 0.0 && buffer <= 500.0 ? 1 : 0) != 0, (String)"Buffer [%s] is out of [0 500] km range", (Object)buffer);
        this.initBufferedRegion(line, buffer);
    }

    public Region(Region region) {
        Preconditions.checkNotNull((Object)region, (Object)"Supplied Region is null");
        this.name = region.name;
        this.border = region.border.clone();
        this.area = (Area)region.area.clone();
        if (region.interiors != null) {
            this.interiors = new ArrayList();
            for (LocationList interior : region.interiors) {
                this.interiors.add(interior.clone());
            }
        }
    }

    public boolean contains(Location loc) {
        return this.area.contains(loc.getLongitude(), loc.getLatitude());
    }

    public boolean contains(Region region) {
        Area areaUnion = (Area)this.area.clone();
        areaUnion.add(region.area);
        return this.area.equals(areaUnion);
    }

    public boolean isRectangular() {
        return this.area.isRectangular();
    }

    public void addInterior(Region region) {
        Region.validateRegion(region);
        Preconditions.checkArgument((boolean)this.contains(region), (Object)"Region must completely contain supplied interior Region");
        LocationList newInterior = region.border.clone();
        Area newArea = Region.createArea(newInterior);
        if (this.interiors != null) {
            for (LocationList interior : this.interiors) {
                Area existing = Region.createArea(interior);
                existing.intersect(newArea);
                Preconditions.checkArgument((boolean)existing.isEmpty(), (Object)"Supplied interior Region overlaps existing interiors");
            }
        } else {
            this.interiors = new ArrayList();
        }
        this.interiors.add(newInterior.unmodifiableList());
        this.area.subtract(region.area);
    }

    public List<LocationList> getInteriors() {
        return this.interiors != null ? Collections.unmodifiableList(this.interiors) : null;
    }

    public LocationList getBorder() {
        return this.border.unmodifiableList();
    }

    public Area getShape() {
        return (Area)this.area.clone();
    }

    public double getExtent() {
        Rectangle2D rRect = this.area.getBounds2D();
        Location origin = new Location(rRect.getCenterY(), rRect.getCenterX());
        ArrayList<Double> xs = new ArrayList<Double>();
        ArrayList<Double> ys = new ArrayList<Double>();
        for (Location loc : this.border) {
            LocationVector v = LocationUtils.vector(origin, loc);
            double az = v.getAzimuthRad();
            double d = v.getHorzDistance();
            xs.add(Math.sin(az) * d);
            ys.add(Math.cos(az) * d);
        }
        xs.add((Double)xs.get(0));
        ys.add((Double)ys.get(0));
        double area = Region.computeArea(Doubles.toArray(xs), Doubles.toArray(ys));
        if (this.interiors != null) {
            for (LocationList interiorBorder : this.interiors) {
                xs.clear();
                ys.clear();
                for (Location loc : interiorBorder) {
                    LocationVector v = LocationUtils.vector(origin, loc);
                    double az = v.getAzimuthRad();
                    double d = v.getHorzDistance();
                    xs.add(Math.sin(az) * d);
                    ys.add(Math.cos(az) * d);
                }
                xs.add((Double)xs.get(0));
                ys.add((Double)ys.get(0));
                area -= Region.computeArea(Doubles.toArray(xs), Doubles.toArray(ys));
            }
        }
        return area;
    }

    public static void main(String[] args) throws IOException {
        Region region = new Region(new Location(34.0, -118.0), new Location(36.0, -120.0));
        region.setName("Simple region");
        System.out.println(region.toFeature().toJSON());
        GriddedRegion gridReg = new GriddedRegion(region, 0.5, null);
        gridReg.setName("Example gridded region");
        System.out.println(gridReg.toFeature().toJSON());
    }

    private static double computeArea(double[] xs, double[] ys) {
        Region.positivize(xs);
        Region.positivize(ys);
        double area = 0.0;
        for (int i = 0; i < xs.length - 1; ++i) {
            area += xs[i] * ys[i + 1] - xs[i + 1] * ys[i];
        }
        return Math.abs(area) / 2.0;
    }

    private static void positivize(double[] v) {
        double min = Doubles.min((double[])v);
        if (min >= 0.0) {
            return;
        }
        min = Math.abs(min);
        int i = 0;
        while (i < v.length) {
            int n = i++;
            v[n] = v[n] + min;
        }
    }

    public boolean equalsRegion(Region r) {
        return this.area.equals(r.area);
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof Region)) {
            return false;
        }
        Region r = (Region)obj;
        if (!this.getName().equals(r.getName())) {
            return false;
        }
        return this.equalsRegion(r);
    }

    public int hashCode() {
        return this.border.hashCode() ^ this.name.hashCode();
    }

    public Region clone() {
        return new Region(this);
    }

    public double getMinLat() {
        return this.area.getBounds2D().getMinY();
    }

    public double getMaxLat() {
        return this.area.getBounds2D().getMaxY();
    }

    public double getMinLon() {
        return this.area.getBounds2D().getMinX();
    }

    public double getMaxLon() {
        return this.area.getBounds2D().getMaxX();
    }

    public double distanceToLocation(Location loc) {
        Preconditions.checkNotNull((Object)loc, (Object)"Supplied location is null");
        if (this.contains(loc)) {
            return 0.0;
        }
        double min = this.border.minDistToLine(loc);
        double temp = Math.abs(LocationUtils.distanceToLineSegmentFast((Location)this.border.get(this.border.size() - 1), (Location)this.border.get(0), loc));
        return Math.min(temp, min);
    }

    @Override
    public String getName() {
        return this.name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String toString() {
        String str = "Region\n\tMinLat: " + this.getMinLat() + "\n\tMinLon: " + this.getMinLon() + "\n\tMaxLat: " + this.getMaxLat() + "\n\tMaxLon: " + this.getMaxLon();
        return str;
    }

    @Override
    public Element toXMLMetadata(Element root) {
        return this.toXMLMetadata(root, XML_METADATA_NAME);
    }

    public Element toXMLMetadata(Element root, String elemName) {
        Element xml = root.addElement(elemName);
        xml = this.border.toXMLMetadata(xml);
        String name = this.name;
        if (name == null) {
            name = "";
        }
        xml.addAttribute("name", name);
        if (this.type != null) {
            xml.addAttribute("borderType", this.type.name());
        }
        return root;
    }

    public static Region fromXMLMetadata(Element e) {
        String name;
        LocationList list = LocationList.fromXMLMetadata(e.element("LocationList"));
        BorderType type = BorderType.MERCATOR_LINEAR;
        Attribute typeAtt = e.attribute("borderType");
        if (typeAtt != null) {
            type = BorderType.valueOf(typeAtt.getValue());
        }
        Region region = new Region(list, type);
        Attribute nameAtt = e.attribute("name");
        if (nameAtt != null && (name = nameAtt.getValue()).length() > 0) {
            region.setName(name);
        }
        return region;
    }

    public static Region getGlobalRegion() {
        LocationList gll = new LocationList();
        gll.add(new Location(-90.0, -180.0));
        gll.add(new Location(-90.0, 180.0));
        gll.add(new Location(90.0, 180.0));
        gll.add(new Location(90.0, -180.0));
        return new Region(gll, BorderType.MERCATOR_LINEAR);
    }

    public static Region intersect(Region r1, Region r2) {
        Region.validateRegion(r1);
        Region.validateRegion(r2);
        Area newArea = (Area)r1.area.clone();
        newArea.intersect(r2.area);
        if (newArea.isEmpty()) {
            return null;
        }
        Region newRegion = new Region();
        newRegion.area = newArea;
        newRegion.border = Region.createBorder(newArea, true);
        return newRegion;
    }

    public boolean intersects(Region other) {
        Region.validateRegion(this);
        Region.validateRegion(other);
        Area newArea = (Area)this.area.clone();
        newArea.intersect(other.area);
        return !newArea.isEmpty();
    }

    public static Region[] subtract(Region minuend, Region subtrahend) {
        Area newArea = (Area)minuend.area.clone();
        newArea.intersect(subtrahend.area);
        if (newArea.isEmpty()) {
            return null;
        }
        newArea = (Area)minuend.area.clone();
        newArea.subtract(subtrahend.area);
        if (newArea.isEmpty()) {
            return new Region[0];
        }
        ArrayList<Area> solids = new ArrayList<Area>();
        ArrayList<Area> holes = new ArrayList<Area>();
        Region.splitArea(newArea, solids, holes);
        if (solids.isEmpty()) {
            if (!holes.isEmpty()) {
                ArrayList<Region> holeRegions = new ArrayList<Region>();
                for (Area area : holes) {
                    Region hole = new Region();
                    hole.area = area;
                    hole.border = Region.createBorder(area, true);
                    holeRegions.add(hole);
                }
                if (!Region.areHolesSignificant(minuend, holeRegions)) {
                    holes.clear();
                }
            }
            Preconditions.checkState((boolean)holes.isEmpty(), (String)"Have 0 solids but %s holes", (int)holes.size());
            return new Region[0];
        }
        ArrayList<Region> regions = new ArrayList<Region>();
        for (Area area : solids) {
            Region region = new Region();
            region.area = area;
            region.border = Region.createBorder(area, true);
            if (region.border.isEmpty()) continue;
            regions.add(region);
        }
        for (Area area : holes) {
            Region hole = new Region();
            hole.area = area;
            hole.border = Region.createBorder(area, true);
            if (hole.border.isEmpty()) continue;
            boolean found = false;
            for (Region region : regions) {
                if (!region.contains(hole)) continue;
                found = true;
                region.addInterior(hole);
                break;
            }
            if (!found && !Region.isHoleSignificant(minuend, hole)) {
                found = true;
            }
            Preconditions.checkState((boolean)found, (Object)"No solid region found which contains hole!");
        }
        Region[] ret = regions.toArray(new Region[0]);
        return ret;
    }

    private static boolean isHoleSignificant(Region minuend, Region hole) {
        return Region.areHolesSignificant(minuend, Lists.newArrayList((Object[])new Region[]{hole}));
    }

    private static boolean areHolesSignificant(Region minuend, List<Region> holes) {
        double minuendExtent = minuend.getExtent();
        double holeExtent = 0.0;
        for (Region hole : holes) {
            if (hole.border.isEmpty()) continue;
            holeExtent += hole.getExtent();
        }
        double ratio = holeExtent / minuendExtent;
        return ratio > 0.001 && holeExtent > 1.0;
    }

    private static void splitArea(Area multiArea, List<Area> solids, List<Area> holes) {
        Region.splitArea(multiArea, solids, holes, 0);
    }

    private static void splitArea(Area multiArea, List<Area> solids, List<Area> holes, int curRecursion) {
        if (curRecursion == 20) {
            throw new IllegalStateException("Failed to recursively split area into non-singular solids and holes after " + curRecursion + " iterations");
        }
        PathIterator iter = multiArea.getPathIterator(null);
        Path2D.Double poly = new Path2D.Double(0);
        double[] moveToPt = null;
        double[] prevPoint = null;
        double directionTest = 0.0;
        int index = 0;
        while (!iter.isDone()) {
            double[] point = new double[6];
            int type = iter.currentSegment(point);
            if (type == 0) {
                poly.moveTo(point[0], point[1]);
                moveToPt = point;
            } else if (type == 4) {
                if (moveToPt != null) {
                    directionTest += (moveToPt[0] - prevPoint[0]) * (moveToPt[1] + prevPoint[1]);
                }
                boolean hole = directionTest < 0.0;
                poly.closePath();
                Area area = new Area(poly);
                if (!area.isEmpty()) {
                    if (!area.isSingular()) {
                        Region.splitArea(area, solids, holes, curRecursion + 1);
                    } else if (hole) {
                        holes.add(area);
                    } else {
                        solids.add(area);
                    }
                }
                ++index;
                poly = new Path2D.Double(0);
            } else if (type == 1) {
                poly.lineTo(point[0], point[1]);
                if (prevPoint != null) {
                    directionTest += (point[0] - prevPoint[0]) * (point[1] + prevPoint[1]);
                }
            } else {
                throw new IllegalStateException("Unexpected area path type: " + type);
            }
            iter.next();
            prevPoint = point;
        }
    }

    public static Region union(Region r1, Region r2) {
        Region.validateRegion(r1);
        Region.validateRegion(r2);
        Area newArea = (Area)r1.area.clone();
        newArea.add(r2.area);
        if (!newArea.isSingular()) {
            return null;
        }
        Region newRegion = new Region();
        newRegion.area = newArea;
        newRegion.border = Region.createBorder(newArea, true);
        return newRegion;
    }

    private static void validateRegion(Region r) {
        Preconditions.checkNotNull((Object)r, (Object)"Supplied Region is null");
        Preconditions.checkArgument((boolean)r.area.isSingular(), (Object)"Region must be singular");
    }

    private static Area createArea(LocationList border) {
        Area area = new Area(border.toPath());
        if (!area.isSingular()) {
            ArrayList<Area> solids = new ArrayList<Area>();
            ArrayList<Area> holes = new ArrayList<Area>();
            Region.splitArea(area, solids, holes);
            boolean hasRealHole = false;
            for (Area hole : holes) {
                hasRealHole = hasRealHole || !hole.isEmpty();
            }
            if (!hasRealHole && solids.size() == 1) {
                area = (Area)solids.get(0);
            }
        }
        Preconditions.checkArgument((!area.isEmpty() ? 1 : 0) != 0, (Object)"Internally computed Area is empty");
        Preconditions.checkArgument((boolean)area.isSingular(), (Object)"Internally computed Area is not a single closed path");
        return area;
    }

    private void initBorderedRegion(LocationList border, BorderType type) {
        this.type = type;
        int lastIndex = border.size() - 1;
        if (((Location)border.get(lastIndex)).equals(border.get(0))) {
            border.remove(lastIndex);
        }
        if (type.equals((Object)BorderType.GREAT_CIRCLE)) {
            LocationList gcBorder = new LocationList();
            Location start = (Location)border.get(border.size() - 1);
            for (int i = 0; i < border.size(); ++i) {
                gcBorder.add(start);
                Location end = (Location)border.get(i);
                double distance = LocationUtils.horzDistance(start, end);
                while (distance > 100.0) {
                    double azRad = LocationUtils.azimuthRad(start, end);
                    Location segLoc = LocationUtils.location(start, azRad, 100.0);
                    gcBorder.add(segLoc);
                    start = segLoc;
                    distance = LocationUtils.horzDistance(start, end);
                }
                start = end;
            }
            this.border = gcBorder.clone();
        } else {
            this.border = border.clone();
        }
        this.area = Region.createArea(this.border);
    }

    private void initCircularRegion(Location center, double radius) {
        this.border = Region.createLocationCircle(center, radius);
        this.area = Region.createArea(this.border);
    }

    private void initBufferedRegion(LocationList line, double buffer) {
        this.area = new Area();
        Location prevLoc = null;
        for (Location loc : line) {
            if (this.area.isEmpty()) {
                this.area.add(Region.createArea(Region.createLocationCircle(loc, buffer)));
                prevLoc = loc;
                continue;
            }
            this.area.add(Region.createArea(Region.createLocationBox(prevLoc, loc, buffer)));
            this.area.add(Region.createArea(Region.createLocationCircle(loc, buffer)));
            prevLoc = loc;
        }
        this.border = Region.createBorder(this.area, true);
    }

    private static LocationList createBorder(Area area, boolean clean) {
        PathIterator pi = area.getPathIterator(null);
        LocationList ll = new LocationList();
        double[] vertex = new double[6];
        while (!pi.isDone()) {
            int type = pi.currentSegment(vertex);
            double lon = vertex[0];
            double lat = vertex[1];
            if (type != 4) {
                ll.add(new Location(lat, lon));
            }
            pi.next();
        }
        if (clean) {
            LocationList llClean = new LocationList();
            Location prev = (Location)ll.get(ll.size() - 1);
            for (Location loc : ll) {
                if (loc.equals(prev)) continue;
                llClean.add(loc);
                prev = loc;
            }
            ll = llClean;
        }
        return ll;
    }

    private static LocationList createLocationCircle(Location center, double radius) {
        LocationList ll = new LocationList();
        for (double angle = 0.0; angle < 360.0; angle += 10.0) {
            ll.add(LocationUtils.location(center, angle * GeoTools.TO_RAD, radius));
        }
        return ll;
    }

    private static LocationList createLocationBox(Location p1, Location p2, double distance) {
        double az12 = LocationUtils.azimuthRad(p1, p2);
        double az21 = LocationUtils.azimuthRad(p2, p1);
        LocationList ll = new LocationList();
        ll.add(LocationUtils.location(p1, az12 - 1.5707963267948966, distance));
        ll.add(LocationUtils.location(p1, az12 + 1.5707963267948966, distance));
        ll.add(LocationUtils.location(p2, az21 - 1.5707963267948966, distance));
        ll.add(LocationUtils.location(p2, az21 + 1.5707963267948966, distance));
        return ll;
    }

    private void writeObject(ObjectOutputStream os) throws IOException {
        os.writeObject(this.name);
        os.writeObject(this.border);
        os.writeObject(this.interiors);
    }

    private void readObject(ObjectInputStream is) throws IOException, ClassNotFoundException {
        this.name = (String)is.readObject();
        this.border = (LocationList)is.readObject();
        this.interiors = (ArrayList)is.readObject();
        this.area = Region.createArea(this.border);
        if (this.interiors != null) {
            for (LocationList interior : this.interiors) {
                Area intArea = Region.createArea(interior);
                this.area.subtract(intArea);
            }
        }
    }

    public Feature toFeature() {
        String name = this.getName();
        if (name != null && name.equals(NAME_DEFAULT)) {
            name = null;
        }
        return new Feature(name, (Geometry)new Geometry.Polygon(this), new FeatureProperties());
    }

    public static Region fromFeature(Feature feature) {
        Preconditions.checkNotNull((Object)feature.geometry, (Object)"Feature is missing geometry");
        Geometry geometry = feature.geometry;
        if (feature.geometry instanceof Geometry.GeometryCollection) {
            geometry = null;
            for (Geometry oGeom : ((Geometry.GeometryCollection)feature.geometry).geometries) {
                if (oGeom instanceof Geometry.Polygon || oGeom instanceof Geometry.MultiPolygon) {
                    Preconditions.checkState((geometry == null ? 1 : 0) != 0, (Object)"Multiple polygons found for region, a region must be supplied as either a single Polygon or a single Polgon within a MultiPolygon geometry");
                    geometry = oGeom;
                    continue;
                }
                if (!(oGeom instanceof Geometry.Point) && !(oGeom instanceof Geometry.MultiPoint)) continue;
                try {
                    return GriddedRegion.fromFeature(feature);
                }
                catch (Exception e) {
                    System.err.println("WARNING: Region feature contains geometry of type " + String.valueOf((Object)oGeom.type) + ", tried and failed load it as a GriddedRegion: " + e.getMessage());
                }
            }
            Preconditions.checkState((geometry != null ? 1 : 0) != 0, (Object)"GeometryCollection specified, but no Polygon or MultiPolygon geometries found within.");
        }
        Preconditions.checkState((geometry instanceof Geometry.Polygon || geometry instanceof Geometry.MultiPolygon ? 1 : 0) != 0, (String)"Unexpected geometry type for Region: %s", (Object)((Object)geometry.type));
        if (geometry instanceof Geometry.MultiPolygon) {
            List<Region> list = ((Geometry.MultiPolygon)geometry).asRegions();
            Preconditions.checkState((list.size() == 1 ? 1 : 0) != 0, (String)"Must have exactly 1 polygon, have %s", (int)list.size());
            return list.get(0);
        }
        Region region = ((Geometry.Polygon)geometry).asRegion();
        if (feature.id != null) {
            region.setName(feature.id.toString());
        }
        return region;
    }

    public static class Adapter
    extends TypeAdapter<Region> {
        private Feature.FeatureAdapter featureAdapter = new Feature.FeatureAdapter();

        public void write(JsonWriter out, Region value) throws IOException {
            if (value == null) {
                out.nullValue();
            } else {
                this.featureAdapter.write(out, value.toFeature());
            }
        }

        public Region read(JsonReader in) throws IOException {
            Feature feature = this.featureAdapter.read(in);
            if (feature == null) {
                return null;
            }
            return Region.fromFeature(feature);
        }
    }
}

