/*
 * Decompiled with CFR 0.152.
 */
package org.opensha.commons.data.comcat.plot;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.io.Files;
import gov.usgs.earthquake.event.EventQuery;
import gov.usgs.earthquake.event.EventWebService;
import java.awt.Color;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import javax.imageio.ImageIO;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.MissingOptionException;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.opensha.commons.calc.magScalingRelations.magScalingRelImpl.WC1994_MagLengthRelationship;
import org.opensha.commons.data.CSVFile;
import org.opensha.commons.data.comcat.ComcatAccessor;
import org.opensha.commons.data.comcat.ComcatRegion;
import org.opensha.commons.data.comcat.ComcatRegionAdapter;
import org.opensha.commons.data.comcat.plot.ComcatDataPlotter;
import org.opensha.commons.data.function.DefaultXY_DataSet;
import org.opensha.commons.data.function.EvenlyDiscretizedFunc;
import org.opensha.commons.geo.Location;
import org.opensha.commons.geo.LocationList;
import org.opensha.commons.geo.LocationUtils;
import org.opensha.commons.geo.LocationVector;
import org.opensha.commons.geo.Region;
import org.opensha.commons.gui.plot.PlotCurveCharacterstics;
import org.opensha.commons.gui.plot.PlotLineType;
import org.opensha.commons.param.Parameter;
import org.opensha.commons.util.ClassUtils;
import org.opensha.commons.util.ComparablePairing;
import org.opensha.commons.util.ExceptionUtils;
import org.opensha.commons.util.FileNameUtils;
import org.opensha.commons.util.FileUtils;
import org.opensha.commons.util.MarkdownUtils;
import org.opensha.sha.earthquake.faultSysSolution.FaultSystemSolution;
import org.opensha.sha.earthquake.observedEarthquake.ObsEqkRupture;
import org.opensha.sha.faultSurface.FaultSection;
import org.opensha.sha.faultSurface.RuptureSurface;
import scratch.UCERF3.enumTreeBranches.FaultModels;
import scratch.UCERF3.erf.ETAS.ETAS_CatalogIO;
import scratch.UCERF3.erf.ETAS.ETAS_EqkRupture;
import scratch.UCERF3.erf.ETAS.analysis.ETAS_AbstractPlot;
import scratch.UCERF3.erf.ETAS.analysis.ETAS_ComcatComparePlot;
import scratch.UCERF3.erf.ETAS.analysis.ETAS_FaultParticipationPlot;
import scratch.UCERF3.erf.ETAS.analysis.ETAS_MFD_Plot;
import scratch.UCERF3.erf.ETAS.launcher.ETAS_Config;
import scratch.UCERF3.erf.ETAS.launcher.ETAS_Launcher;
import scratch.UCERF3.erf.ETAS.launcher.util.ETAS_CatalogIteration;

public class ComcatReportPageGen {
    private ComcatAccessor accessor;
    private ObsEqkRupture mainshock;
    private Region region;
    private double minFetchMag;
    private double daysBefore;
    private static final double MIN_FETCH_MAG_DEFAULT = 0.0;
    private static final double DAYS_BEFORE_DEFAULT = 3.0;
    private static final double min_radius = 10.0;
    private double radius;
    private long originTime;
    private String placeName;
    private List<ObsEqkRupture> foreshocks;
    private List<ObsEqkRupture> aftershocks;
    private ComcatDataPlotter plotter;
    private EventWebService service;
    private Collection<FaultSection> faults;
    private ETAS_Config etasRun;
    private File etasOutputDir;
    private static DecimalFormat percentProbDF = new DecimalFormat("0.000%");
    private static final DecimalFormat optionalDigitDF = new DecimalFormat("0.##");

    public ComcatReportPageGen(String eventID, Region region, double minFetchMag, double daysBefore) {
        ComcatAccessor accessor = new ComcatAccessor();
        ObsEqkRupture mainshock = accessor.fetchEvent(eventID, false, true);
        this.init(accessor, mainshock, region, minFetchMag, daysBefore);
    }

    public ComcatReportPageGen(String eventID, double radius, double minFetchMag, double daysBefore) {
        ComcatAccessor accessor = new ComcatAccessor();
        ObsEqkRupture mainshock = accessor.fetchEvent(eventID, false, true);
        this.radius = radius;
        Region region = new Region(mainshock.getHypocenterLocation(), radius);
        this.init(accessor, mainshock, region, minFetchMag, daysBefore);
    }

    public ComcatReportPageGen(String eventID, double minFetchMag, double daysBefore) {
        ComcatAccessor accessor = new ComcatAccessor();
        ObsEqkRupture mainshock = accessor.fetchEvent(eventID, false, true);
        this.radius = ComcatReportPageGen.getDefaultRadius(mainshock.getMag());
        Region region = new Region(mainshock.getHypocenterLocation(), this.radius);
        this.init(accessor, mainshock, region, minFetchMag, daysBefore);
    }

    public ComcatReportPageGen(CommandLine cmd) throws IOException {
        ComcatAccessor accessor = new ComcatAccessor();
        String eventID = cmd.getOptionValue("event-id");
        ObsEqkRupture mainshock = accessor.fetchEvent(eventID, false, true);
        this.radius = cmd.hasOption("radius") ? Double.parseDouble(cmd.getOptionValue("radius")) : ComcatReportPageGen.getDefaultRadius(mainshock.getMag());
        Region region = new Region(mainshock.getHypocenterLocation(), this.radius);
        double minFetchMag = cmd.hasOption("min-mag") ? Double.parseDouble(cmd.getOptionValue("min-mag")) : 0.0;
        double daysBefore = cmd.hasOption("days-before") ? Double.parseDouble(cmd.getOptionValue("days-before")) : 3.0;
        this.init(accessor, mainshock, region, minFetchMag, daysBefore);
        if (cmd.hasOption("etas-dir")) {
            File etasDir = new File(cmd.getOptionValue("etas-dir"));
            System.out.println("Loading UCERF3-ETAS from: " + etasDir.getAbsolutePath());
            ETAS_Config config = ETAS_Config.readJSON(new File(etasDir, "config.json"));
            this.addETAS(config);
            if (cmd.hasOption("etas-output-dir")) {
                this.etasOutputDir = new File(cmd.getOptionValue("etas-output-dir"));
                Preconditions.checkState((this.etasOutputDir.exists() || this.etasOutputDir.mkdir() ? 1 : 0) != 0);
                this.etasOutputDir = new File(this.etasOutputDir, eventID);
                Preconditions.checkState((this.etasOutputDir.exists() || this.etasOutputDir.mkdir() ? 1 : 0) != 0);
            }
        }
    }

    private static double getDefaultRadius(double mag) {
        double radius = new WC1994_MagLengthRelationship().getMedianLength(mag);
        System.out.println("WC 1994 Radius: " + (float)radius);
        radius *= 2.0;
        if (radius < 10.0) {
            System.out.println("Reverting to min radius of 10.0");
            radius = 10.0;
        }
        return radius;
    }

    private void init(ComcatAccessor accessor, ObsEqkRupture mainshock, Region region, double minFetchMag, double daysBefore) {
        double maxMag;
        ComcatRegion cReg;
        this.accessor = accessor;
        this.mainshock = mainshock;
        this.region = region;
        this.minFetchMag = minFetchMag;
        this.daysBefore = daysBefore;
        System.out.println("Mainshock is a M" + (float)mainshock.getMag());
        System.out.println("\tHypocenter: " + String.valueOf(mainshock.getHypocenterLocation()));
        this.originTime = mainshock.getOriginTime();
        this.placeName = null;
        for (Parameter param : Lists.newArrayList(mainshock.getAddedParametersIterator())) {
            if (!param.getName().equals("description")) continue;
            this.placeName = param.getValue().toString();
            break;
        }
        System.out.println("Place name: " + this.placeName);
        double minDepth = -10.0;
        double maxDepth = Math.max(30.0, 2.0 * mainshock.getHypocenterLocation().getDepth());
        ComcatRegion comcatRegion = cReg = region instanceof ComcatRegion ? (ComcatRegion)((Object)region) : new ComcatRegionAdapter(region);
        if (daysBefore > 0.0) {
            long startTime = this.originTime - (long)(8.64E7 * daysBefore);
            System.out.println("Fetching " + (float)daysBefore + " days of foreshocks");
            this.foreshocks = accessor.fetchEventList(mainshock.getEventId(), startTime, this.originTime, minDepth, maxDepth, cReg, false, false, minFetchMag);
            maxMag = Double.NEGATIVE_INFINITY;
            for (ObsEqkRupture rup : this.foreshocks) {
                maxMag = Math.max(maxMag, rup.getMag());
            }
            System.out.println("Found " + this.foreshocks.size() + " foreshocks, maxMag=" + maxMag);
        }
        long endTime = System.currentTimeMillis();
        System.out.println("Fetching aftershocks");
        this.aftershocks = accessor.fetchEventList(mainshock.getEventId(), this.originTime, endTime, minDepth, maxDepth, cReg, false, false, minFetchMag);
        maxMag = Double.NEGATIVE_INFINITY;
        for (ObsEqkRupture rup : this.aftershocks) {
            maxMag = Math.max(maxMag, rup.getMag());
        }
        System.out.println("Found " + this.aftershocks.size() + " aftershocks, maxMag=" + maxMag);
        this.plotter = new ComcatDataPlotter(mainshock, this.originTime, endTime, this.foreshocks, this.aftershocks);
        this.service = accessor.getEventWebService();
    }

    public String generateDirName() {
        SimpleDateFormat df = new SimpleDateFormat("yyyy_MM_dd");
        Object name = df.format(new Date(this.originTime));
        name = (String)name + "-" + this.mainshock.getEventId() + "-M" + optionalDigitDF.format(this.mainshock.getMag());
        String placeStripped = FileNameUtils.simplify(this.placeName);
        name = (String)name + "-" + placeStripped;
        return name;
    }

    public void addETAS(ETAS_Config config) {
        this.etasRun = config;
    }

    public void generateReport(File outputDir, String executiveSummary) throws IOException {
        double delta;
        double mag;
        File resourcesDir = new File(outputDir, "resources");
        Preconditions.checkState((resourcesDir.exists() || resourcesDir.mkdir() ? 1 : 0) != 0);
        ArrayList<String> lines = new ArrayList<String>();
        lines.add("# " + optionalDigitDF.format(this.mainshock.getMag()) + ", " + this.placeName);
        lines.add("");
        if (executiveSummary != null && !executiveSummary.isEmpty()) {
            lines.add(executiveSummary);
            lines.add("");
        }
        int tocIndex = lines.size();
        String topLink = "*[(top)](#table-of-contents)*";
        lines.add("");
        lines.add("## Mainshock Details");
        lines.add(topLink);
        lines.add("");
        lines.addAll(this.generateDetailLines(this.mainshock, resourcesDir, "##", topLink));
        lines.add("## Sequence Details");
        lines.add(topLink);
        lines.add("");
        SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss z");
        df.setTimeZone(TimeZone.getTimeZone("UTC"));
        Object line = "These plots show the aftershock sequence, using data sourced from [ComCat](https://earthquake.usgs.gov/data/comcat/). They were last updated at " + df.format(new Date(this.plotter.getEndTime())) + ", ";
        long deltaMillis = this.plotter.getEndTime() - this.originTime;
        double deltaSecs = (double)deltaMillis / 1000.0;
        double deltaMins = deltaSecs / 60.0;
        double deltaHours = deltaMins / 60.0;
        double deltaDays = deltaHours / 24.0;
        line = deltaDays > 2.0 ? (String)line + optionalDigitDF.format(deltaDays) + " days" : (deltaHours > 2.0 ? (String)line + optionalDigitDF.format(deltaHours) + " hours" : (String)line + optionalDigitDF.format(deltaMins) + " minutes");
        line = (String)line + " after the mainshock.";
        lines.add((String)line);
        lines.add("");
        line = this.aftershocks.size() + " M&ge;" + optionalDigitDF.format(this.minFetchMag) + " earthquakes";
        if (this.radius > 0.0) {
            line = (String)line + " within " + optionalDigitDF.format(this.radius) + " km of the mainshock's epicenter.";
        } else {
            double maxDist = 0.0;
            for (Location location : this.region.getBorder()) {
                maxDist = Math.max(maxDist, LocationUtils.horzDistanceFast(location, this.mainshock.getHypocenterLocation()));
            }
            line = (String)line + " within a custom region that lies within a circle with radius " + optionalDigitDF.format(this.radius) + " km from the mainshock's epicenter.";
        }
        lines.add((String)line);
        lines.add("");
        if (this.aftershocks.size() > 0) {
            this.writeEventCSV(this.aftershocks, new File(resourcesDir, "aftershocks.csv"));
            double maxAftershock = this.minFetchMag;
            for (ObsEqkRupture obsEqkRupture : this.aftershocks) {
                maxAftershock = Math.max(maxAftershock, obsEqkRupture.getMag());
            }
            double minFloor = Math.floor(this.minFetchMag);
            double maxFloor = Math.floor(maxAftershock);
            int num = (int)(Math.round(maxFloor - minFloor) + 1L);
            EvenlyDiscretizedFunc magFunc = new EvenlyDiscretizedFunc(Math.floor(this.minFetchMag) + 0.5, num, 1.0);
            MarkdownUtils.TableBuilder table = MarkdownUtils.tableBuilder();
            table.initNewLine();
            table.addColumn("");
            ArrayList<Long> maxOTs = new ArrayList<Long>();
            long hour = 3600000L;
            if (deltaHours > 1.0) {
                table.addColumn("First Hour");
                maxOTs.add(this.originTime + hour);
            }
            long day = hour * 24L;
            if (deltaDays >= 1.0) {
                table.addColumn("First Day");
                maxOTs.add(this.originTime + day);
            }
            if (deltaDays >= 7.0) {
                table.addColumn("First Week");
                maxOTs.add(this.originTime + 7L * day);
            }
            if (deltaDays >= 30.0) {
                table.addColumn("First Month");
                maxOTs.add(this.originTime + 30L * day);
            }
            table.addColumn("To Date");
            maxOTs.add(this.plotter.getEndTime());
            table.finalizeLine();
            int[][] counts = new int[magFunc.size()][maxOTs.size()];
            for (ObsEqkRupture obsEqkRupture : this.aftershocks) {
                int m = magFunc.getClosestXIndex(obsEqkRupture.getMag());
                long ot = obsEqkRupture.getOriginTime();
                for (int mi = 0; mi <= m; ++mi) {
                    for (int oi = 0; oi < maxOTs.size(); ++oi) {
                        if (ot > (Long)maxOTs.get(oi)) continue;
                        int[] nArray = counts[mi];
                        int n = oi;
                        nArray[n] = nArray[n] + 1;
                    }
                }
            }
            for (int mi = 0; mi < magFunc.size(); ++mi) {
                table.initNewLine();
                table.addColumn("**M " + (int)magFunc.getX(mi) + "**");
                for (int count : counts[mi]) {
                    table.addColumn(count);
                }
                table.finalizeLine();
            }
            lines.add("");
            lines.addAll(table.build());
            lines.add("### Magnitude Vs. Time Plot");
            lines.add(topLink);
            lines.add("");
            line = "This plot shows the magnitude vs. time evolution of the sequence. The mainshock is ploted as a brown circle";
            if (this.plotter.getForeshocks() != null && !this.plotter.getForeshocks().isEmpty()) {
                line = (String)line + ", foreshocks are plotted as magenta circles";
            }
            line = (String)line + ", and aftershocks are plotted as cyan circles.";
            lines.add((String)line);
            lines.add("");
            ArrayList<String> magTimeTitles = new ArrayList<String>();
            ArrayList<String> arrayList = new ArrayList<String>();
            if (deltaDays > 7.0) {
                magTimeTitles.add("First Week");
                String prefix = "aftershocks_mag_vs_time_week";
                arrayList.add(prefix);
                this.plotter.plotMagTimeFunc(resourcesDir, prefix, "Magnitude Vs. Time", null, 7.0, null);
            }
            if (deltaDays > 30.0) {
                magTimeTitles.add("First Month");
                String prefix = "aftershocks_mag_vs_time_month";
                arrayList.add(prefix);
                this.plotter.plotMagTimeFunc(resourcesDir, prefix, "Magnitude Vs. Time", null, 30.0, null);
            }
            magTimeTitles.add("To Date");
            String fullPrefix = "aftershocks_mag_vs_time";
            arrayList.add(fullPrefix);
            this.plotter.plotMagTimeFunc(resourcesDir, fullPrefix, "Magnitude Vs. Time", null, deltaDays, null);
            if (magTimeTitles.size() > 1) {
                table = MarkdownUtils.tableBuilder();
                table.addLine(magTimeTitles);
                table.initNewLine();
                for (String prefix : arrayList) {
                    table.addColumn("![Mag vs Time Plot](resources/" + prefix + ".png)");
                }
                table.finalizeLine();
                lines.addAll(table.build());
            } else {
                lines.add("![Mag vs Time Plot](resources/" + fullPrefix + ".png)");
            }
            lines.add("");
            lines.add("### Aftershock Locations");
            lines.add(topLink);
            lines.add("");
            line = "Map view of the aftershock sequence, plotted as cyan circles. The mainshock ";
            line = this.plotter.getForeshocks() != null && !this.plotter.getForeshocks().isEmpty() ? (String)line + " and foreshocks are plotted below in brown and magenta circles respectively" : (String)line + " is plotted below as a brown circle";
            line = (String)line + ", but may be obscured by aftershocks. Nearby UCERF3 fault traces are plotted in gray lines, and the region used to fetch aftershock data in a dashed dark gray line.";
            lines.add((String)line);
            lines.add("");
            if (deltaDays > 1.0) {
                table = MarkdownUtils.tableBuilder();
                table.initNewLine();
                table.addColumn("First Day");
                if (deltaDays > 7.0) {
                    table.addColumn("First Week");
                }
                table.addColumn("To Date");
                table.finalizeLine();
                table.initNewLine();
                this.plotMap(resourcesDir, "map_first_day", " ", this.originTime + 86400000L);
                table.addColumn("![First Day](resources/map_first_day.png)");
                if (deltaDays > 7.0) {
                    this.plotMap(resourcesDir, "map_first_week", " ", this.originTime + 604800000L);
                    table.addColumn("![First Day](resources/map_first_week.png)");
                }
                this.plotMap(resourcesDir, "map_to_date", " ", this.plotter.getEndTime());
                table.addColumn("![First Day](resources/map_to_date.png)");
                table.finalizeLine();
                lines.addAll(table.build());
            } else {
                this.plotMap(resourcesDir, "map_to_date", " ", this.plotter.getEndTime());
                lines.add("![First Day](resources/map_to_date.png)");
            }
            lines.add("");
            lines.add("### Cumulative Number Plot");
            lines.add(topLink);
            lines.add("");
            lines.add("This plot shows the cumulative number of M&ge;" + optionalDigitDF.format(this.minFetchMag) + " aftershocks as a function of time since the mainshock.");
            lines.add("");
            this.plotter.plotTimeFuncPlot(resourcesDir, "aftershocks_vs_time", this.minFetchMag);
            lines.add("![Time Func](resources/aftershocks_vs_time.png)");
            lines.add("");
            lines.add("### Magnitude-Number Distributions (MNDs)");
            lines.add(topLink);
            lines.add("");
            lines.add("These plot shows the magnitude-number distribution of the aftershock sequence thus far. The left plot gives an incremental distribution (the count in each magnitude bin), and the right plot a cumulative distribution (the count in or above each magnitude bin).");
            lines.add("");
            this.plotter.plotMagNumPlot(resourcesDir, "aftershocks_mag_num_incremental", false, this.minFetchMag, this.minFetchMag, false, false, null);
            this.plotter.plotMagNumPlot(resourcesDir, "aftershocks_mag_num_cumulative", true, this.minFetchMag, this.minFetchMag, false, false, null);
            table = MarkdownUtils.tableBuilder();
            table.addLine("Incremental MND", "Cumulative MND");
            table.addLine("![Incremental](resources/aftershocks_mag_num_incremental.png)", "![Cumulative](resources/aftershocks_mag_num_cumulative.png)");
            lines.addAll(table.build());
            lines.add("");
        }
        double sigMag = Math.max(4.0, Math.min(6.0, this.mainshock.getMag() - 1.0));
        if (this.foreshocks != null) {
            ArrayList<ObsEqkRupture> sigForeshocks = new ArrayList<ObsEqkRupture>();
            for (ObsEqkRupture foreshock : this.foreshocks) {
                mag = foreshock.getMag();
                if (!(mag >= sigMag)) continue;
                sigForeshocks.add(foreshock);
            }
            this.writeEventCSV(this.foreshocks, new File(resourcesDir, "foreshocks.csv"));
            if (!sigForeshocks.isEmpty()) {
                lines.add("## Significant Foreshocks");
                lines.add(topLink);
                lines.add("");
                lines.add("Foreshock(s) with M&ge;6 or with M&ge;M<sub>Mainshock</sub>-1.");
                lines.add("");
                for (ObsEqkRupture rup : sigForeshocks) {
                    delta = (double)(this.originTime - rup.getOriginTime()) / 8.64E7;
                    double mag2 = rup.getMag();
                    lines.add("### M" + optionalDigitDF.format(mag2) + " " + optionalDigitDF.format(delta) + " days before");
                    lines.add(topLink);
                    lines.add("");
                    lines.addAll(this.generateDetailLines(rup, resourcesDir, "###", topLink));
                    lines.add("");
                }
            }
        }
        ArrayList<ObsEqkRupture> sigAftershocks = new ArrayList<ObsEqkRupture>();
        for (ObsEqkRupture rup : this.aftershocks) {
            mag = rup.getMag();
            if (!(mag >= sigMag)) continue;
            sigAftershocks.add(rup);
        }
        if (!sigAftershocks.isEmpty()) {
            lines.add("## Significant Aftershocks");
            lines.add(topLink);
            lines.add("");
            lines.add("Aftershocks(s) with M&ge;6 or with M&ge;M<sub>Mainshock</sub>-1.");
            lines.add("");
            for (ObsEqkRupture rup : sigAftershocks) {
                delta = (double)(rup.getOriginTime() - this.originTime) / 8.64E7;
                double mag3 = rup.getMag();
                lines.add("### M" + optionalDigitDF.format(mag3) + " " + optionalDigitDF.format(delta) + " days after");
                lines.add(topLink);
                lines.add("");
                lines.addAll(this.generateDetailLines(rup, resourcesDir, "###", topLink));
            }
            lines.add("");
        }
        if (this.etasRun != null) {
            lines.addAll(this.generateETASLines(this.etasRun, true, resourcesDir, "#", topLink));
        }
        ArrayList<String> arrayList = new ArrayList<String>();
        arrayList.add("## Table Of Contents");
        arrayList.add("");
        arrayList.addAll(MarkdownUtils.buildTOC(lines, 2, 3));
        lines.addAll(tocIndex, arrayList);
        MarkdownUtils.writeReadmeAndHTML(lines, outputDir);
    }

    private void writeEventCSV(List<ObsEqkRupture> events, File csvFile) throws IOException {
        CSVFile<String> csv = new CSVFile<String>(true);
        csv.addLine("Origin Time (epoch ms)", "ComCat Event ID", "Magnitude", "Latitude", "Longitude", "Depth (km)");
        for (ObsEqkRupture event : events) {
            Location hypo = event.getHypocenterLocation();
            csv.addLine("" + event.getOriginTime(), event.getEventId(), "" + (float)event.getMag(), "" + (float)hypo.getLatitude(), "" + (float)hypo.getLongitude(), "" + (float)hypo.getDepth());
        }
        csv.writeToFile(csvFile);
    }

    private List<String> generateDetailLines(ObsEqkRupture event, File resourcesDir, String curHeading, String topLink) throws IOException {
        List events;
        String eventID = event.getEventId();
        EventQuery query = new EventQuery();
        query.setEventId(eventID);
        try {
            events = this.service.getEvents(query);
            Preconditions.checkState((!events.isEmpty() ? 1 : 0) != 0, (Object)"Event not found");
        }
        catch (Exception e) {
            System.err.println("Could not retrieve event '" + eventID + "' from Comcat");
            throw ExceptionUtils.asRuntimeException(e);
        }
        Preconditions.checkState((events.size() == 1 ? 1 : 0) != 0, (Object)("More that 1 match? " + events.size()));
        JSONObject obj = (JSONObject)events.get(0);
        JSONObject prop = (JSONObject)obj.get((Object)"properties");
        JSONObject prods = (JSONObject)prop.get((Object)"products");
        ArrayList<String> lines = new ArrayList<String>();
        String url = prop.get((Object)"url").toString();
        System.out.println("URL: " + url);
        lines.add("Information and plots in the section are taken from the [USGS event page](" + url + "), accessed through ComCat.");
        lines.add("");
        MarkdownUtils.TableBuilder table = MarkdownUtils.tableBuilder();
        SimpleDateFormat df = new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss z");
        SimpleDateFormat zoneDF = new SimpleDateFormat("z");
        Date date = new Date(this.originTime);
        TimeZone local = df.getTimeZone();
        df.setTimeZone(TimeZone.getTimeZone("UTC"));
        zoneDF.setTimeZone(df.getTimeZone());
        table.addLine("Field", "Value");
        table.addLine("Magnitude", optionalDigitDF.format(event.getMag()) + " (" + String.valueOf(prop.get((Object)"magType")) + ")");
        table.addLine("Time (" + zoneDF.format(date) + ")", df.format(date));
        df.setTimeZone(local);
        zoneDF.setTimeZone(local);
        table.addLine("Time (" + zoneDF.format(date) + ")", df.format(date));
        Location hypo = event.getHypocenterLocation();
        table.addLine("Location", (float)hypo.getLatitude() + ", " + (float)hypo.getLongitude());
        table.addLine("Depth", (float)hypo.getDepth() + " km");
        table.addLine("Status", prop.get((Object)"status"));
        lines.addAll(table.build());
        String shakemapImageURL = null;
        JSONArray shakemaps = (JSONArray)prods.get((Object)"shakemap");
        if (shakemaps != null && shakemaps.size() > 0) {
            JSONObject shakemap = (JSONObject)shakemaps.get(0);
            shakemapImageURL = ComcatReportPageGen.fetchShakeMapImage(shakemap);
            System.out.println("Shakemap image: " + shakemapImageURL);
        }
        String dyfiImageURL = null;
        JSONArray dyfis = (JSONArray)prods.get((Object)"dyfi");
        if (dyfis != null && dyfis.size() > 0) {
            JSONObject dyfi = (JSONObject)dyfis.get(0);
            dyfiImageURL = ComcatReportPageGen.fetchDYFIImage(dyfi);
            System.out.println("DYFI image: " + dyfiImageURL);
        }
        String[] pagerImageURLs = null;
        JSONArray pagers = (JSONArray)prods.get((Object)"losspager");
        if (pagers != null && pagers.size() > 0) {
            JSONObject pager = (JSONObject)pagers.get(0);
            pagerImageURLs = ComcatReportPageGen.fetchPagerImage(pager);
            System.out.println("Pager images: " + pagerImageURLs[0] + " " + pagerImageURLs[1]);
        }
        String mechImageURL = null;
        JSONArray mechs = (JSONArray)prods.get((Object)"moment-tensor");
        if (mechs != null && mechs.size() > 0) {
            JSONObject mech = (JSONObject)mechs.get(0);
            mechImageURL = ComcatReportPageGen.fetchMechImage(mech);
            System.out.println("Mech image: " + mechImageURL);
        }
        if (shakemapImageURL != null || dyfiImageURL != null || pagerImageURLs != null || mechImageURL != null) {
            lines.add("");
            lines.add(curHeading + "# USGS Products");
            lines.add(topLink);
            lines.add("");
            table = MarkdownUtils.tableBuilder();
            table.initNewLine();
            if (shakemapImageURL != null) {
                table.addColumn("<center>**[ShakeMap](" + url + "/shakemap/)**</center>");
            }
            if (dyfiImageURL != null) {
                table.addColumn("<center>**[Did You Feel It?](" + url + "/dyfi/)**</center>");
            }
            if (pagerImageURLs != null) {
                table.addColumn("<center>**[PAGER](" + url + "/pager/)**</center>");
            }
            if (mechImageURL != null) {
                table.addColumn("<center>**[Moment Tensor](" + url + "/moment-tensor/)**</center>");
            }
            table.finalizeLine();
            table.initNewLine();
            if (shakemapImageURL != null) {
                File smFile = new File(resourcesDir, eventID + "_shakemap.jpg");
                FileUtils.downloadURL(shakemapImageURL, smFile);
                table.addColumn("![ShakeMap](resources/" + smFile.getName() + ")");
            }
            if (dyfiImageURL != null) {
                File dyfiFile = new File(resourcesDir, eventID + "_dyfi.jpg");
                FileUtils.downloadURL(dyfiImageURL, dyfiFile);
                table.addColumn("![DYFI](resources/" + dyfiFile.getName() + ")");
            }
            if (pagerImageURLs != null) {
                File fatalFile = new File(resourcesDir, eventID + "_pager_fatalities.png");
                File econFile = new File(resourcesDir, eventID + "_pager_economic.png");
                FileUtils.downloadURL(pagerImageURLs[0], fatalFile);
                FileUtils.downloadURL(pagerImageURLs[1], econFile);
                File combined = new File(resourcesDir, eventID + "_pager.png");
                ComcatReportPageGen.combinePager(fatalFile, econFile, combined);
                table.addColumn("![PEGER](resources/" + combined.getName() + ")");
            }
            if (mechImageURL != null) {
                File mechFile = new File(resourcesDir, eventID + "_mechanism.jpg");
                FileUtils.downloadURL(mechImageURL, mechFile);
                table.addColumn("![Mechanism](resources/" + mechFile.getName() + ")");
            }
            table.finalizeLine();
            lines.addAll(table.wrap(3, 0).build());
        }
        double distThreshold = 10.0;
        HashMap<String, Double> faultDists = new HashMap<String, Double>();
        for (FaultSection fault : this.getU3Faults()) {
            RuptureSurface surf = fault.getFaultSurface(1.0);
            double minDist = Double.POSITIVE_INFINITY;
            for (Location loc : surf.getEvenlyDiscritizedListOfLocsOnSurface()) {
                double dist = LocationUtils.linearDistanceFast(loc, hypo);
                minDist = Double.min(minDist, dist);
            }
            if (!(minDist < distThreshold)) continue;
            faultDists.put(fault.getName(), minDist);
        }
        lines.add("");
        lines.add(curHeading + "# Nearby Faults");
        lines.add(topLink);
        lines.add("");
        lines.add("");
        Object line = faultDists.isEmpty() ? "No UCERF3 fault sections are" : (faultDists.size() == 1 ? "1 UCERF3 fault section is" : faultDists.size() + " UCERF3 fault sections are");
        line = (String)line + " within " + optionalDigitDF.format(distThreshold) + "km of this event's hypocenter";
        line = faultDists.isEmpty() ? (String)line + "." : (String)line + ":";
        lines.add((String)line);
        lines.add("");
        if (!faultDists.isEmpty()) {
            for (String name : ComparablePairing.getSortedData(faultDists)) {
                lines.add("* " + name + ": " + optionalDigitDF.format(faultDists.get(name)) + "km");
            }
        }
        return lines;
    }

    private static String fetchShakeMapImage(JSONObject shakemap) {
        JSONObject contents = (JSONObject)shakemap.get((Object)"contents");
        if (contents == null) {
            return null;
        }
        JSONObject intensityOBJ = (JSONObject)contents.get((Object)"download/intensity.jpg");
        if (intensityOBJ == null) {
            return null;
        }
        return (String)intensityOBJ.get((Object)"url");
    }

    private static String fetchDYFIImage(JSONObject dyfi) {
        JSONObject contents = (JSONObject)dyfi.get((Object)"contents");
        if (contents == null) {
            return null;
        }
        for (Object key : contents.keySet()) {
            if (!key.toString().trim().endsWith("_ciim.jpg")) continue;
            JSONObject intensityOBJ = (JSONObject)contents.get(key);
            if (intensityOBJ == null) {
                return null;
            }
            return (String)intensityOBJ.get((Object)"url");
        }
        return null;
    }

    private static String[] fetchPagerImage(JSONObject dyfi) {
        JSONObject contents = (JSONObject)dyfi.get((Object)"contents");
        if (contents == null) {
            return null;
        }
        JSONObject fatalOBJ = (JSONObject)contents.get((Object)"alertfatal.png");
        JSONObject econOBJ = (JSONObject)contents.get((Object)"alertecon.png");
        if (fatalOBJ == null || econOBJ == null) {
            return null;
        }
        return new String[]{(String)fatalOBJ.get((Object)"url"), (String)econOBJ.get((Object)"url")};
    }

    private static void combinePager(File top, File bottom, File output) throws IOException {
        int x;
        int y;
        BufferedImage topIMG = ImageIO.read(top);
        BufferedImage botIMG = ImageIO.read(bottom);
        int width = Integer.max(topIMG.getWidth(), botIMG.getWidth());
        int height = topIMG.getHeight() + botIMG.getHeight();
        BufferedImage comb = new BufferedImage(width, height, topIMG.getType());
        for (y = 0; y < topIMG.getHeight(); ++y) {
            for (x = 0; x < topIMG.getWidth(); ++x) {
                comb.setRGB(x, y, topIMG.getRGB(x, y));
            }
        }
        for (y = 0; y < topIMG.getHeight(); ++y) {
            for (x = 0; x < topIMG.getWidth(); ++x) {
                comb.setRGB(x, y + topIMG.getHeight(), botIMG.getRGB(x, y));
            }
        }
        ImageIO.write((RenderedImage)comb, "png", output);
    }

    private static String fetchMechImage(JSONObject mech) {
        JSONObject contents = (JSONObject)mech.get((Object)"contents");
        if (contents == null) {
            return null;
        }
        for (Object key : contents.keySet()) {
            if (!key.toString().trim().endsWith("_mechanism.jpg")) continue;
            JSONObject mechOBJ = (JSONObject)contents.get(key);
            if (mechOBJ == null) {
                return null;
            }
            return (String)mechOBJ.get((Object)"url");
        }
        return null;
    }

    private static void printJSON(JSONObject json) {
        ComcatReportPageGen.printJSON(json, "");
    }

    private static void printJSON(JSONObject json, String prefix) {
        for (Object key : json.keySet()) {
            Object val = json.get(key);
            if (val != null && val.toString().startsWith("[{")) {
                String str = val.toString();
                try {
                    val = new JSONParser().parse(str.substring(1, str.length() - 1));
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            if (val != null && val instanceof JSONObject) {
                System.out.println(prefix + String.valueOf(key) + ":");
                Object prefix2 = prefix;
                if (prefix2 == null) {
                    prefix2 = "";
                }
                prefix2 = (String)prefix2 + "\t";
                ComcatReportPageGen.printJSON((JSONObject)val, (String)prefix2);
                continue;
            }
            System.out.println(prefix + String.valueOf(key) + ": " + String.valueOf(val));
        }
    }

    private synchronized Collection<FaultSection> getU3Faults() {
        if (this.faults == null) {
            Map<Integer, FaultSection> map = FaultModels.FM3_1.getFaultSectionIDMap();
            map = new HashMap<Integer, FaultSection>(map);
            map.putAll(FaultModels.FM3_2.getFaultSectionIDMap());
            this.faults = map.values();
        }
        return this.faults;
    }

    private void plotMap(File resourcesDir, String prefix, String title, long endTime) throws IOException {
        ArrayList<ObsEqkRupture> events = new ArrayList<ObsEqkRupture>();
        for (ObsEqkRupture rup : this.aftershocks) {
            if (rup.getOriginTime() > endTime) continue;
            events.add(rup);
        }
        double padding = 30.0;
        double fRad = 1.5;
        if (this.radius > 0.0) {
            padding = Math.max(padding, fRad * this.radius);
        } else {
            LocationList border = this.region.getBorder();
            for (int i = 0; i < border.size(); ++i) {
                Location l1 = (Location)border.get(i);
                for (int j = i + 1; j < border.size(); ++j) {
                    Location l2 = (Location)border.get(j);
                    padding = Math.max(padding, fRad * LocationUtils.horzDistanceFast(l1, l2));
                }
            }
        }
        Location ll = new Location(this.region.getMinLat(), this.region.getMinLon());
        ll = LocationUtils.location(ll, new LocationVector(225.0, padding, 0.0));
        Location ul = new Location(this.region.getMaxLat(), this.region.getMaxLon());
        ul = LocationUtils.location(ul, new LocationVector(45.0, padding, 0.0));
        Region mapRegion = new Region(ul, ll);
        ArrayList<DefaultXY_DataSet> inputFuncs = new ArrayList<DefaultXY_DataSet>();
        ArrayList<PlotCurveCharacterstics> inputChars = new ArrayList<PlotCurveCharacterstics>();
        PlotCurveCharacterstics faultChar = new PlotCurveCharacterstics(PlotLineType.SOLID, 2.0f, Color.GRAY);
        boolean firstFault = true;
        for (FaultSection fault : this.getU3Faults()) {
            DefaultXY_DataSet xy = new DefaultXY_DataSet();
            for (Location loc : fault.getFaultTrace()) {
                xy.set(loc.getLongitude(), loc.getLatitude());
            }
            if (firstFault) {
                xy.setName("Faults");
                firstFault = false;
            }
            inputFuncs.add(xy);
            inputChars.add(faultChar);
        }
        DefaultXY_DataSet regXY = new DefaultXY_DataSet();
        LocationList border = this.region.getBorder();
        for (int i = 0; i <= border.size(); ++i) {
            Location loc = (Location)border.get(i % border.size());
            regXY.set(loc.getLongitude(), loc.getLatitude());
        }
        regXY.setName("Data Region");
        inputFuncs.add(regXY);
        inputChars.add(new PlotCurveCharacterstics(PlotLineType.DASHED, 3.0f, Color.DARK_GRAY));
        this.plotter.plotMap(resourcesDir, prefix, title, mapRegion, events, this.minFetchMag, inputFuncs, inputChars);
    }

    private List<String> generateETASLines(final ETAS_Config config, final boolean snapToNow, File resourcesDir, String curHeading, String topLink) throws IOException {
        String date;
        ArrayList<String> lines = new ArrayList<String>();
        String title = "UCERF3-ETAS Forecast";
        Object description = "This section gives results from the UCERF3-ETAS short-term forecasting model. This model is described in [Field et al. (2017)](http://bssa.geoscienceworld.org/lookup/doi/10.1785/0120160173), and computes probabilities of this sequence triggering subsequent aftershocks, including events on known faults.";
        long deltaMillis = config.getSimulationStartTimeMillis() - this.originTime;
        Preconditions.checkState((deltaMillis >= 0L ? 1 : 0) != 0, (Object)"ETAS forecast starts before mainshock");
        double deltaDays = (double)deltaMillis / 8.64E7;
        double deltaYears = (double)deltaMillis / 3.15576E10;
        String deltaStr = ETAS_AbstractPlot.getTimeLabel(deltaYears, true).toLowerCase();
        SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss z");
        if (snapToNow) {
            config.setStartTimeMillis(this.plotter.getEndTime());
            date = df.format(new Date(this.plotter.getEndTime()));
            description = (String)description + "\n\nProbabilities are inherantly time-dependent. Those stated here are for time periods beginning the instant when this report was generated, " + date + ".";
        } else {
            date = df.format(new Date(config.getSimulationStartTimeMillis()));
            description = (String)description + "\n\nProbabilities are inherantly time-dependent. Those stated here are for time periods beginning when the model was run, " + date + ".";
        }
        description = deltaDays > 0.041666666666666664 ? (String)description + " The model was updated with all observed aftershcoks up to " + deltaStr + " after the mainshock, and may be out of date, especially if large aftershocks have occurred subsequently or a significant amount of time has passed since the last update." : (String)description + " The model has not been updated with any observed aftershocks and may be out of date, especially if large aftershock have occurred subsequently or a significant amount of time has passed since the mainshock.";
        lines.add(curHeading + "# " + title);
        lines.add(topLink);
        lines.add("");
        lines.add((String)description);
        lines.add("");
        lines.add("Results are summarized below and should be considered preliminary. The exact timing, size, location, or number of aftershocks cannot be predicted, and all probabilities are uncertain.");
        lines.add("");
        ETAS_Config.ComcatMetadata meta = config.getComcatMetadata();
        ETAS_Config.ComcatMetadata oMeta = new ETAS_Config.ComcatMetadata(this.region, meta.eventID, meta.minDepth, meta.maxDepth, meta.minMag, meta.startTime, meta.endTime);
        oMeta.magComplete = oMeta.magComplete;
        config.setComcatMetadata(oMeta);
        File inputFile = null;
        for (ETAS_Config.BinaryFilteredOutputConfig filter : config.getBinaryOutputFilters()) {
            File file = new File(config.getOutputDir(), filter.getPrefix() + ".bin");
            if (!file.exists()) {
                file = new File(config.getOutputDir(), filter.getPrefix() + "_partial.bin");
            }
            if (!file.exists()) continue;
            inputFile = file;
            System.out.println("Simulation input file: " + String.valueOf(inputFile));
            break;
        }
        Preconditions.checkNotNull(inputFile, (Object)"input not found");
        ETAS_Launcher launcher = new ETAS_Launcher(config, false);
        final FaultSystemSolution fss = launcher.checkOutFSS();
        final ArrayList<ETAS_AbstractPlot> plots = new ArrayList<ETAS_AbstractPlot>();
        ETAS_ComcatComparePlot comcatPlot = new ETAS_ComcatComparePlot(config, launcher);
        plots.add(comcatPlot);
        String mfdPrefix = "etas_mfd";
        ETAS_MFD_Plot mfdPlot = new ETAS_MFD_Plot(config, launcher, mfdPrefix, false, true);
        plots.add(mfdPlot);
        String faultPrefix = "etas_fault_prefix";
        ETAS_FaultParticipationPlot faultPlot = new ETAS_FaultParticipationPlot(config, launcher, faultPrefix, false, true);
        plots.add(faultPlot);
        boolean filterSpontaneous = false;
        for (ETAS_AbstractPlot plot : plots) {
            filterSpontaneous = filterSpontaneous || plot.isFilterSpontaneous();
        }
        final boolean isFilterSpontaneous = filterSpontaneous;
        System.out.println("Processing " + config.getSimulationName());
        int numProcessed = ETAS_CatalogIteration.processCatalogs(inputFile, new ETAS_CatalogIteration.Callback(){
            final /* synthetic */ ComcatReportPageGen this$0;
            {
                this.this$0 = this$0;
            }

            @Override
            public void processCatalog(ETAS_CatalogIO.ETAS_Catalog catalog, int index) {
                if (snapToNow) {
                    long minOT = config.getSimulationStartTimeMillis();
                    ETAS_CatalogIO.ETAS_Catalog modCatalog = new ETAS_CatalogIO.ETAS_Catalog(catalog.getSimulationMetadata());
                    for (int i = 0; i < catalog.size(); ++i) {
                        ETAS_EqkRupture rup = (ETAS_EqkRupture)catalog.get(i);
                        if (((ETAS_EqkRupture)catalog.get(i)).getOriginTime() < minOT) continue;
                        modCatalog.add(rup);
                    }
                    catalog = modCatalog;
                }
                ETAS_CatalogIO.ETAS_Catalog triggeredOnlyCatalog = null;
                if (isFilterSpontaneous) {
                    triggeredOnlyCatalog = ETAS_Launcher.getFilteredNoSpontaneous(config, catalog);
                }
                for (ETAS_AbstractPlot plot : plots) {
                    plot.processCatalog(catalog, triggeredOnlyCatalog, fss);
                }
            }
        }, -1, 0.0);
        System.out.println("Processed " + numProcessed + " catalogs");
        ExecutorService exec = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
        ArrayList futures = new ArrayList();
        System.out.println("Finalizing plots...");
        for (ETAS_AbstractPlot eTAS_AbstractPlot : plots) {
            List<? extends Runnable> runnables = eTAS_AbstractPlot.finalize(resourcesDir, fss, exec);
            if (runnables == null) continue;
            for (Runnable runnable : runnables) {
                futures.add(exec.submit(runnable));
            }
        }
        System.out.println("Waiting on " + futures.size() + " futures...");
        for (Future future : futures) {
            try {
                future.get();
            }
            catch (InterruptedException | ExecutionException e) {
                throw ExceptionUtils.asRuntimeException(e);
            }
        }
        System.out.println("DONE finalizing");
        exec.shutdown();
        launcher.checkInFSS(fss);
        MarkdownUtils.TableBuilder table = MarkdownUtils.tableBuilder();
        double[] dArray = mfdPlot.getDurations();
        ETAS_MFD_Plot.MFD_Stats[] mfdStats = mfdPlot.getFullStats();
        HashSet<Double> mfdIncludeDurations = new HashSet<Double>();
        mfdIncludeDurations.add(0.019164955509924708);
        mfdIncludeDurations.add(0.08213552361396304);
        table.initNewLine();
        table.addColumn("");
        for (double duration : dArray) {
            if (!mfdIncludeDurations.contains(duration)) continue;
            table.addColumn(ETAS_MFD_Plot.getTimeLabel(duration, false));
        }
        table.finalizeLine();
        EvenlyDiscretizedFunc evenlyDiscretizedFunc = mfdStats[0].getProbFunc(0);
        for (int m = 0; m < evenlyDiscretizedFunc.size(); ++m) {
            double mag = evenlyDiscretizedFunc.getX(m);
            if (mag != Math.floor(mag)) continue;
            table.initNewLine();
            table.addColumn("**M&ge;" + optionalDigitDF.format(mag) + "**");
            double maxProb = 0.0;
            for (int d = 0; d < dArray.length; ++d) {
                double duration = dArray[d];
                if (!mfdIncludeDurations.contains(duration)) continue;
                double prob = mfdStats[d].getProbFunc(0).getY(m);
                maxProb = Math.max(prob, maxProb);
                table.addColumn(ComcatReportPageGen.getProbStr(prob));
            }
            table.finalizeLine();
            if (maxProb < 1.0E-4) break;
        }
        lines.add("");
        lines.add("This table gives forecasted one week and one month probabilities for events triggered by this sequence; it does not include the long-term probability of such events.");
        lines.add("");
        lines.addAll(table.build());
        lines.add("");
        lines.add(curHeading + "## ETAS Forecasted Magnitude Vs. Time");
        lines.add(topLink);
        lines.add("");
        Object line = "These plots show the show the magnitude versus time probability function since simulation start. Observed event data lie on top, with those input to the simulation plotted as magenta circles and those that occurred after the simulation start time as cyan circles. Time is relative to ";
        if (this.plotter.getMainshock() == null) {
            line = (String)line + "the simulation start time.";
        } else {
            ObsEqkRupture mainshock = this.plotter.getMainshock();
            Double mag = mainshock.getMag();
            line = (String)line + "the mainshock (M" + optionalDigitDF.format(mag) + ", " + mainshock.getEventId() + ", plotted as a brown circle).";
        }
        line = (String)line + " Probabilities are only shown above the minimum simulated magnitude, M=2.5.";
        lines.add((String)line);
        table = MarkdownUtils.tableBuilder();
        HashMap<String[], Double> magTimeDurations = new HashMap<String[], Double>();
        if (!snapToNow) {
            magTimeDurations.put(new String[]{"To Date", "mag_time_full.png"}, comcatPlot.getCurDuration());
        }
        magTimeDurations.put(new String[]{"One Week", "mag_time_week.png"}, 0.019164955509924708);
        magTimeDurations.put(new String[]{"One Month", "mag_time_month.png"}, 0.08213552361396304);
        List<String[]> sortedDurations = ComparablePairing.getSortedData(magTimeDurations);
        File etasResourcesDir = null;
        if (this.etasOutputDir != null) {
            etasResourcesDir = new File(this.etasOutputDir, "resources");
            Preconditions.checkState((etasResourcesDir.exists() || etasResourcesDir.mkdir() ? 1 : 0) != 0);
            for (String[] key : magTimeDurations.keySet()) {
                Files.copy((File)new File(resourcesDir, key[1]), (File)new File(etasResourcesDir, key[1]));
            }
        }
        table.initNewLine();
        for (String[] label : sortedDurations) {
            table.addColumn(label[0]);
        }
        table.finalizeLine();
        table.initNewLine();
        for (String[] fName : sortedDurations) {
            table.addColumn("![Mag-time plot](resources/" + fName[1] + ")");
        }
        table.finalizeLine();
        lines.add("");
        lines.addAll(table.build());
        lines.add("");
        lines.add(curHeading + "## ETAS Spatial Distribution Forecast");
        lines.add(topLink);
        lines.add("");
        lines.add("These plots show the predicted spatial distribution of aftershocks above the given magnitude threshold and for the given time period. The 'Current' plot shows the forecasted spatial distribution to date, along with as any observed aftershocks overlaid with cyan circles. Observed aftershocks will be included in the week/month plots as well if the forecasted time window has elapsed.");
        lines.add("");
        table = MarkdownUtils.tableBuilder();
        HashSet<Double> includeDurations = new HashSet<Double>();
        includeDurations.add(0.019164955509924708);
        includeDurations.add(0.08213552361396304);
        if (!snapToNow) {
            includeDurations.add(comcatPlot.getCurDuration());
        }
        table.initNewLine();
        table.addColumn("");
        for (double duration : comcatPlot.getDurations()) {
            if (!includeDurations.contains(duration)) continue;
            table.addColumn(comcatPlot.getMapTableLabel(duration));
        }
        table.finalizeLine();
        double[] durations = comcatPlot.getDurations();
        double[] minMags = comcatPlot.getMinMags();
        HashSet<Double> includeMags = new HashSet<Double>();
        double minAboveZero = Double.POSITIVE_INFINITY;
        for (double mag : minMags) {
            if (!(mag > 0.0)) continue;
            minAboveZero = Math.min(minAboveZero, mag);
        }
        includeMags.add(minAboveZero);
        includeMags.add(5.0);
        String[][] mapPrefixes = comcatPlot.getMapProbPrefixes();
        for (int m = 0; m < minMags.length; ++m) {
            double mag = minMags[m];
            if (!includeMags.contains(mag)) continue;
            table.initNewLine();
            table.addColumn("**M&ge;" + optionalDigitDF.format(mag) + "**");
            for (int d = 0; d < durations.length; ++d) {
                if (!includeDurations.contains(durations[d])) continue;
                table.addColumn("![Map](resources/" + mapPrefixes[d][m] + ".png)");
                if (etasResourcesDir == null) continue;
                Files.copy((File)new File(resourcesDir, mapPrefixes[d][m] + ".png"), (File)new File(etasResourcesDir, mapPrefixes[d][m] + ".png"));
            }
            table.finalizeLine();
        }
        lines.addAll(table.build());
        lines.add("");
        lines.add(curHeading + "## ETAS Fault Trigger Probabilities");
        lines.add(topLink);
        lines.add("");
        lines.add("The table below summarizes the probabilities of this sequence triggering large supra-seismogenic aftershocks on nearby known active faults.");
        lines.add("");
        table = MarkdownUtils.tableBuilder();
        double[] faultMags = new double[]{0.0, 7.0};
        table.initNewLine();
        table.addColumn("Fault Section");
        double[] faultDurations = faultPlot.getDurations();
        for (double mag : faultMags) {
            String magStr = mag > 0.0 ? "M&ge;" + optionalDigitDF.format(mag) : "supra-seis";
            for (double duration : faultDurations) {
                if (!includeDurations.contains(duration)) continue;
                String durStr = ETAS_AbstractPlot.getTimeShortLabel(duration);
                table.addColumn(durStr + " " + magStr + " prob");
            }
        }
        table.finalizeLine();
        Map<Integer, ETAS_FaultParticipationPlot.FaultStats> faultStats = faultPlot.getParentSectStats();
        List<Integer> sortedIDs = ComparablePairing.getSortedData(faultStats);
        if (sortedIDs.size() > 10) {
            sortedIDs = sortedIDs.subList(0, 10);
        }
        for (int parentID : sortedIDs) {
            ETAS_FaultParticipationPlot.FaultStats stats = faultStats.get(parentID);
            stats.getTriggeredCumulativeMPDs();
            table.initNewLine();
            table.addColumn("**" + stats.getName() + "**");
            EvenlyDiscretizedFunc[] mpds = stats.getTriggeredCumulativeMPDs();
            for (double mag : faultMags) {
                for (int d = 0; d < mpds.length; ++d) {
                    if (!includeDurations.contains(faultDurations[d])) continue;
                    double prob = mag > 0.0 ? mpds[d].getInterpolatedY(mag) : mpds[d].getY(0);
                    table.addColumn(ComcatReportPageGen.getProbStr(prob));
                }
            }
            table.finalizeLine();
        }
        lines.addAll(table.build());
        if (this.etasOutputDir != null) {
            MarkdownUtils.writeReadmeAndHTML(lines, this.etasOutputDir);
        }
        return lines;
    }

    private static String getProbStr(double prob) {
        if (prob * 100.0 < 5.0E-4) {
            return "<0.001%";
        }
        return percentProbDF.format(prob);
    }

    private static Options createOptions() {
        Options ops = new Options();
        Option event = new Option("e", "event-id", true, "ComCat event id, e.g. 'ci39126079'");
        event.setRequired(true);
        ops.addOption(event);
        Option minMag = new Option("m", "min-mag", true, "Minimum magnitude of events to fetch (default: 0.0)");
        minMag.setRequired(false);
        ops.addOption(minMag);
        Option daysBefore = new Option("d", "days-before", true, "Number of days of events before the mainshock to fetch (default: 3)");
        daysBefore.setRequired(false);
        ops.addOption(daysBefore);
        Option radius = new Option("r", "radius", true, "Search radius around mainshock for aftershocks. Default is the greater of 10.0 km and twice the Wells & Coppersmith (1994) median rupture length for the mainshock magnitude");
        radius.setRequired(false);
        ops.addOption(radius);
        Option etas = new Option("etas", "etas-dir", true, "Path to a UCERF3-ETAS simulation directory");
        etas.setRequired(false);
        ops.addOption(etas);
        Option etasOutput = new Option("eod", "etas-output-dir", true, "If supplied, ETAS only results will also be written to <path>/<event-id>");
        etasOutput.setRequired(false);
        ops.addOption(etasOutput);
        Option outputDir = new Option("o", "output-dir", true, "Output dirctory. Must supply either this or --output-parent-dir");
        outputDir.setRequired(false);
        ops.addOption(outputDir);
        Option outputParentDir = new Option("opd", "output-parent-dir", true, "Output parent dirctory. The directory name will be generated automatically from the event name, date, and magnitude. Must supply either this or --output-dir");
        outputParentDir.setRequired(false);
        ops.addOption(outputParentDir);
        Option help = new Option("?", "help", false, "Display this message");
        help.setRequired(false);
        ops.addOption(help);
        return ops;
    }

    public static void printHelp(Options options, String appName) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp(120, appName, null, options, null, true);
        System.exit(2);
    }

    public static void printUsage(Options options, String appName) {
        HelpFormatter formatter = new HelpFormatter();
        PrintWriter pw = new PrintWriter(System.out);
        formatter.printUsage(pw, 120, appName, options);
        pw.flush();
        System.exit(2);
    }

    public static void main(String[] args) throws IOException {
        if (args.length == 1 && args[0].equals("--hardcoded")) {
            File mainDir = new File("/home/kevin/git/event-reports");
            Object argStr = "--event-id ci39645386 --min-mag 0d --radius 50";
            argStr = (String)argStr + " --output-parent-dir " + mainDir.getAbsolutePath();
            argStr = (String)argStr + " --etas-dir /home/kevin/OpenSHA/UCERF3/etas/simulations/2023_08_22-ComCatM5p08_ci39645386_1p8DaysAfter_PointSources";
            argStr = (String)argStr + " --etas-output-dir " + mainDir.getAbsolutePath() + "/ucerf3-etas";
            args = Splitter.on((String)" ").splitToList((CharSequence)argStr).toArray(new String[0]);
        }
        try {
            Options options = ComcatReportPageGen.createOptions();
            String appName = ClassUtils.getClassNameWithoutPackage(ComcatReportPageGen.class);
            DefaultParser parser = new DefaultParser();
            if (args.length == 0) {
                ComcatReportPageGen.printUsage(options, appName);
            }
            try {
                CommandLine cmd = parser.parse(options, args);
                if (cmd.hasOption("help") || cmd.hasOption("?")) {
                    ComcatReportPageGen.printHelp(options, appName);
                }
                ComcatReportPageGen pageGen = new ComcatReportPageGen(cmd);
                File outputDir = null;
                if (cmd.hasOption("output-dir")) {
                    Preconditions.checkArgument((!cmd.hasOption("output-parent-dir") ? 1 : 0) != 0, (Object)"Can't supply both --output-dir and --output-parent-dir");
                    outputDir = new File(cmd.getOptionValue("output-dir"));
                } else if (cmd.hasOption("output-parent-dir")) {
                    Preconditions.checkArgument((!cmd.hasOption("output-dir") ? 1 : 0) != 0, (Object)"Can't supply both --output-dir and --output-parent-dir");
                    File parentDir = new File(cmd.getOptionValue("output-parent-dir"));
                    Preconditions.checkState((parentDir.exists() || parentDir.mkdir() ? 1 : 0) != 0, (String)"Output parent dir doesn't exist and can't be created: %s", (Object)parentDir.getAbsolutePath());
                    outputDir = new File(parentDir, pageGen.generateDirName());
                } else {
                    System.err.println("Must supply either --output-dir or --output-parent-dir");
                    ComcatReportPageGen.printUsage(options, appName);
                }
                System.out.println("Output dir: " + outputDir.getAbsolutePath());
                Preconditions.checkState((outputDir.exists() || outputDir.mkdir() ? 1 : 0) != 0);
                pageGen.generateReport(outputDir, null);
            }
            catch (MissingOptionException e) {
                Options helpOps = new Options();
                helpOps.addOption(new Option("h", "help", false, "Display this message"));
                try {
                    CommandLine cmd = parser.parse(helpOps, args);
                    if (cmd.hasOption("help")) {
                        ComcatReportPageGen.printHelp(options, appName);
                    }
                }
                catch (ParseException parseException) {
                    // empty catch block
                }
                System.err.println(e.getMessage());
                ComcatReportPageGen.printUsage(options, appName);
            }
            catch (ParseException e) {
                e.printStackTrace();
                ComcatReportPageGen.printUsage(options, appName);
            }
            System.out.println("Done!");
            System.exit(0);
        }
        catch (Exception e) {
            e.printStackTrace();
            System.exit(1);
        }
    }
}

