/*
 * Decompiled with CFR 0.152.
 */
package org.opensha.commons.geo;

import java.awt.geom.Line2D;
import java.util.Collection;
import org.apache.commons.math3.util.Precision;
import org.opensha.commons.geo.GeoTools;
import org.opensha.commons.geo.Location;
import org.opensha.commons.geo.LocationVector;
import org.opensha.sha.faultSurface.RuptureSurface;

public final class LocationUtils {
    public static final double TOLERANCE = 1.0E-12;

    private LocationUtils() {
    }

    public static double angle(Location p1, Location p2) {
        double lat1 = p1.getLatRad();
        double lat2 = p2.getLatRad();
        double sinDlatBy2 = Math.sin((lat2 - lat1) / 2.0);
        double sinDlonBy2 = Math.sin((p2.getLonRad() - p1.getLonRad()) / 2.0);
        double c = sinDlatBy2 * sinDlatBy2 + Math.cos(lat1) * Math.cos(lat2) * sinDlonBy2 * sinDlonBy2;
        return 2.0 * Math.atan2(Math.sqrt(c), Math.sqrt(1.0 - c));
    }

    public static double horzDistance(Location p1, Location p2) {
        return 6371.0072 * LocationUtils.angle(p1, p2);
    }

    public static double horzDistanceFast(Location p1, Location p2) {
        double lat1 = p1.getLatRad();
        double lat2 = p2.getLatRad();
        double dLat = lat1 - lat2;
        double dLon = (p1.getLonRad() - p2.getLonRad()) * Math.cos((lat1 + lat2) * 0.5);
        return 6371.0072 * Math.sqrt(dLat * dLat + dLon * dLon);
    }

    public static double vertDistance(Location p1, Location p2) {
        return p2.getDepth() - p1.getDepth();
    }

    public static double linearDistance(Location p1, Location p2) {
        double alpha = LocationUtils.angle(p1, p2);
        double R1 = 6371.0072 - p1.getDepth();
        double R2 = 6371.0072 - p2.getDepth();
        double B = R1 * Math.sin(alpha);
        double C = R2 - R1 * Math.cos(alpha);
        return Math.sqrt(B * B + C * C);
    }

    public static double linearDistanceFast(Location p1, Location p2) {
        double h = LocationUtils.horzDistanceFast(p1, p2);
        double v = LocationUtils.vertDistance(p1, p2);
        return Math.sqrt(h * h + v * v);
    }

    public static double cartesianDistance(Location p1, Location p2) {
        return Math.sqrt(LocationUtils.cartesianDistanceSq(p1, p2));
    }

    public static double cartesianDistanceSq(Location p1, Location p2) {
        double latDiff = p1.lat - p2.lat;
        double lonDiff = p1.lon - p2.lon;
        double vDiff = p1.depth - p2.depth;
        return latDiff * latDiff + lonDiff * lonDiff + vDiff * vDiff;
    }

    public static double distanceToLine(Location p1, Location p2, Location p3) {
        double ad13 = LocationUtils.angle(p1, p3);
        double Daz13az12 = LocationUtils.azimuthRad(p1, p3) - LocationUtils.azimuthRad(p1, p2);
        double xtd = Math.asin(Math.sin(ad13) * Math.sin(Daz13az12));
        return Math.abs(xtd) < 1.0E-12 ? 0.0 : xtd * 6371.0072;
    }

    public static double distanceToLineFast(Location p1, Location p2, Location p3) {
        double lat1 = p1.getLatRad();
        double lat2 = p2.getLatRad();
        double lat3 = p3.getLatRad();
        double lon1 = p1.getLonRad();
        double lonScale = Math.cos(0.5 * lat3 + 0.25 * lat1 + 0.25 * lat2);
        double x2 = (p2.getLonRad() - lon1) * lonScale;
        double y2 = lat2 - lat1;
        double x3 = (p3.getLonRad() - lon1) * lonScale;
        double y3 = lat3 - lat1;
        return (x2 * -y3 - -x3 * y2) / Math.sqrt(x2 * x2 + y2 * y2) * 6371.0072;
    }

    public static double distanceToSurf(Location loc, RuptureSurface rupSurf) {
        double minDistance = Double.MAX_VALUE;
        for (Location loc2 : rupSurf.getEvenlyDiscritizedListOfLocsOnSurface()) {
            double vertDist;
            double horzDist = LocationUtils.horzDistance(loc, loc2);
            double totalDist = horzDist * horzDist + (vertDist = LocationUtils.vertDistance(loc, loc2)) * vertDist;
            if (!(totalDist < minDistance)) continue;
            minDistance = totalDist;
        }
        return Math.pow(minDistance, 0.5);
    }

    public static double distanceToSurfFast(Location loc, RuptureSurface rupSurf) {
        double minDistance = Double.MAX_VALUE;
        for (Location loc2 : rupSurf.getEvenlyDiscritizedListOfLocsOnSurface()) {
            double vertDist;
            double horzDist = LocationUtils.horzDistanceFast(loc, loc2);
            double totalDist = horzDist * horzDist + (vertDist = LocationUtils.vertDistance(loc, loc2)) * vertDist;
            if (!(totalDist < minDistance)) continue;
            minDistance = totalDist;
        }
        return Math.pow(minDistance, 0.5);
    }

    public static double distanceToLineSegment(Location p1, Location p2, Location p3) {
        double ad13 = LocationUtils.angle(p1, p3);
        double Daz13az12 = LocationUtils.azimuthRad(p1, p3) - LocationUtils.azimuthRad(p1, p2);
        double xtd = Math.asin(Math.sin(ad13) * Math.sin(Daz13az12));
        double atd = Math.acos(Math.cos(ad13) / Math.cos(xtd)) * 6371.0072;
        if (atd > LocationUtils.horzDistance(p1, p2)) {
            return LocationUtils.horzDistance(p2, p3);
        }
        if (Math.cos(Daz13az12) < 0.0) {
            return LocationUtils.horzDistance(p1, p3);
        }
        return Math.abs(xtd) < 1.0E-12 ? 0.0 : Math.abs(xtd) * 6371.0072;
    }

    public static double distanceToLineSegmentFast(Location p1, Location p2, Location p3) {
        double lat1 = p1.getLatRad();
        double lat2 = p2.getLatRad();
        double lat3 = p3.getLatRad();
        double lon1 = p1.getLonRad();
        double lonScale = Math.cos(0.5 * lat3 + 0.25 * lat1 + 0.25 * lat2);
        double x2 = (p2.getLonRad() - lon1) * lonScale;
        double y2 = lat2 - lat1;
        double x3 = (p3.getLonRad() - lon1) * lonScale;
        double y3 = lat3 - lat1;
        return Line2D.ptSegDist(0.0, 0.0, x2, y2, x3, y3) * 6371.0072;
    }

    public static double azimuthRad(Location p1, Location p2) {
        double lat1 = p1.getLatRad();
        double lat2 = p2.getLatRad();
        if (LocationUtils.isPole(p1)) {
            return lat1 > 0.0 ? Math.PI : 0.0;
        }
        double dLon = p2.getLonRad() - p1.getLonRad();
        double cosLat2 = Math.cos(lat2);
        double azRad = Math.atan2(Math.sin(dLon) * cosLat2, Math.cos(lat1) * Math.sin(lat2) - Math.sin(lat1) * cosLat2 * Math.cos(dLon));
        return (azRad + Math.PI * 2) % (Math.PI * 2);
    }

    public static double azimuth(Location p1, Location p2) {
        return LocationUtils.azimuthRad(p1, p2) * GeoTools.TO_DEG;
    }

    public static Location location(Location p, double azimuth, double distance) {
        return LocationUtils.location(p.getLatRad(), p.getLonRad(), p.getDepth(), azimuth, distance, 0.0);
    }

    public static Location location(Location p, LocationVector d) {
        return LocationUtils.location(p.getLatRad(), p.getLonRad(), p.getDepth(), d.getAzimuth() * GeoTools.TO_RAD, d.getHorzDistance(), d.getVertDistance());
    }

    private static Location location(double lat, double lon, double depth, double az, double dH, double dV) {
        double sinLat1 = Math.sin(lat);
        double cosLat1 = Math.cos(lat);
        double ad = dH / 6371.0072;
        double sinD = Math.sin(ad);
        double cosD = Math.cos(ad);
        double lat2 = Math.asin(sinLat1 * cosD + cosLat1 * sinD * Math.cos(az));
        double lon2 = lon + Math.atan2(Math.sin(az) * sinD * cosLat1, cosD - sinLat1 * Math.sin(lat2));
        return new Location(lat2 * GeoTools.TO_DEG, lon2 * GeoTools.TO_DEG, depth + dV);
    }

    public static LocationVector vector(Location p1, Location p2) {
        LocationVector v = new LocationVector(LocationUtils.azimuth(p1, p2), LocationUtils.horzDistance(p1, p2), LocationUtils.vertDistance(p1, p2));
        return v;
    }

    public static double plunge(Location p1, Location p2) {
        return LocationUtils.vector(p1, p2).getPlunge();
    }

    public static LocationVector bisect(Location p1, Location p2, Location p3) {
        LocationVector v1 = LocationUtils.vector(p2, p1);
        LocationVector v2 = LocationUtils.vector(p2, p3);
        double az = (v2.getAzimuth() + v1.getAzimuth()) / 2.0;
        return new LocationVector(az, 1.0, 0.0);
    }

    public static boolean isPole(Location p) {
        return Math.cos(p.getLatRad()) < 1.0E-12;
    }

    public static boolean areSimilar(Location p1, Location p2) {
        if (!Precision.equals((double)p1.getLatRad(), (double)p2.getLatRad(), (double)1.0E-12)) {
            return false;
        }
        if (!Precision.equals((double)p1.getLonRad(), (double)p2.getLonRad(), (double)1.0E-12)) {
            return false;
        }
        return Precision.equals((double)p1.getDepth(), (double)p2.getDepth(), (double)1.0E-12);
    }

    public static double calcMinLat(Collection<Location> locs) {
        double min = Double.POSITIVE_INFINITY;
        for (Location loc : locs) {
            double val = loc.getLatitude();
            if (!(val < min)) continue;
            min = val;
        }
        return min;
    }

    public static double calcMinLon(Collection<Location> locs) {
        double min = Double.POSITIVE_INFINITY;
        for (Location loc : locs) {
            double val = loc.getLongitude();
            if (!(val < min)) continue;
            min = val;
        }
        return min;
    }

    public static double calcMaxLat(Collection<Location> locs) {
        double max = Double.NEGATIVE_INFINITY;
        for (Location loc : locs) {
            double val = loc.getLatitude();
            if (!(val > max)) continue;
            max = val;
        }
        return max;
    }

    public static double calcMaxLon(Collection<Location> locs) {
        double max = Double.NEGATIVE_INFINITY;
        for (Location loc : locs) {
            double val = loc.getLongitude();
            if (!(val > max)) continue;
            max = val;
        }
        return max;
    }

    public static enum Side {
        RIGHT,
        LEFT,
        ON;

    }
}

