/*
 * Decompiled with CFR 0.152.
 */
package org.opensha.nshmp2.erf.source;

import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.opensha.commons.eq.MagUtils;
import org.opensha.commons.geo.Location;
import org.opensha.nshmp2.erf.source.CH_Data;
import org.opensha.nshmp2.erf.source.FaultERF;
import org.opensha.nshmp2.erf.source.FaultSource;
import org.opensha.nshmp2.erf.source.GR_Data;
import org.opensha.nshmp2.erf.source.MagData;
import org.opensha.nshmp2.erf.source.SourceFile;
import org.opensha.nshmp2.erf.source.SourceMgr;
import org.opensha.nshmp2.erf.source.SourcePredicates;
import org.opensha.nshmp2.util.FaultType;
import org.opensha.nshmp2.util.NSHMP_Utils;
import org.opensha.nshmp2.util.SourceIMR;
import org.opensha.nshmp2.util.SourceRegion;
import org.opensha.nshmp2.util.SourceType;
import org.opensha.sha.faultSurface.FaultTrace;
import org.opensha.sha.magdist.GaussianMagFreqDist;
import org.opensha.sha.magdist.GutenbergRichterMagFreqDist;
import org.opensha.sha.magdist.IncrementalMagFreqDist;
import org.opensha.sha.util.FocalMech;

public class FaultParser {
    static final int TRUNC_TYPE = 2;
    static final int TRUNC_LEVEL = 2;
    private Logger log;

    FaultParser(Logger log) {
        this.log = log;
    }

    FaultERF parse(SourceFile sf) {
        ArrayList srcList = Lists.newArrayList();
        String name = sf.getName();
        SourceRegion srcRegion = sf.getRegion();
        SourceType srcType = sf.getType();
        SourceIMR srcIMR = SourceIMR.imrForSource(srcType, srcRegion, name, null);
        double srcWt = sf.getWeight();
        List<String> dat = sf.readLines();
        int srcNameIdx = srcRegion == SourceRegion.CA || srcRegion == SourceRegion.CEUS ? 3 : 4;
        Iterator<String> srcLines = dat.iterator();
        FaultParser.skipHeader1(srcLines);
        double rMax = NSHMP_Utils.readDouble(srcLines.next(), 1);
        FaultParser.skipHeader2(srcLines);
        MagData md = new MagData(NSHMP_Utils.readLines(srcLines, 4));
        md.toLog(this.log);
        while (srcLines.hasNext()) {
            FaultSource fs = this.createSource(srcLines.next(), sf, srcNameIdx);
            List<String> mfdSrcDat = NSHMP_Utils.readLines(srcLines, fs.nMag);
            this.generateMFDs(fs, mfdSrcDat, md.clone());
            FaultParser.generateFaultTrace(srcLines, fs, this.log);
            FaultParser.toLog(this.log, fs);
            if (fs.mfds.size() == 0) {
                StringBuilder sb = new StringBuilder().append("Source with no mfds");
                FaultParser.appendFaultDat(sb, fs);
                this.log.warning(sb.toString());
            }
            srcList.add(fs);
        }
        if (name.contains("3dip")) {
            this.cleanStrikeSlip(srcList);
        }
        return new FaultERF(name, srcList, srcRegion, srcIMR, srcWt, rMax);
    }

    private void generateMFDs(FaultSource fs, List<String> lines, MagData md) {
        switch (fs.type) {
            case CH: {
                this.buildMFDs_CH(lines, md, fs);
                break;
            }
            case GR: {
                this.buildMFDs_GR(lines, md, fs);
                break;
            }
            case GRB0: {
                this.buildMFDs_GRB0(lines, md, fs);
            }
        }
    }

    private void buildMFDs_CH(List<String> lines, MagData md, FaultSource fs) {
        fs.floats = false;
        for (String line : lines) {
            CH_Data chData = new CH_Data(line);
            this.chBuilder(chData, md, fs);
        }
    }

    private void buildMFDs_GR(List<String> lines, MagData md, FaultSource fs) {
        fs.floats = true;
        ArrayList<GR_Data> grData = new ArrayList<GR_Data>();
        for (String line : lines) {
            grData.add(new GR_Data(line, fs, this.log));
        }
        for (GR_Data gr : grData) {
            if (!gr.hasMagExceptions(this.log, fs, md)) continue;
            md.suppressUncertainty();
        }
        for (GR_Data gr : grData) {
            if (gr.nMag > 1) {
                this.grBuilder(gr, md, fs);
                continue;
            }
            CH_Data ch = new CH_Data(gr.mMin, MagUtils.gr_rate(gr.aVal, gr.bVal, gr.mMin), gr.weight);
            fs.type = FaultType.CH;
            fs.floats = true;
            this.chBuilder(ch, md, fs);
        }
    }

    private void buildMFDs_GRB0(List<String> lines, MagData md, FaultSource fs) {
        fs.floats = true;
        Preconditions.checkArgument((!md.hasAleatory ? 1 : 0) != 0, (String)"Aleatory unc. [%s] is incompatible with GR b=0 branches", (Object)md.aleaSigma);
        ArrayList<GR_Data> grData = new ArrayList<GR_Data>();
        for (String line : lines) {
            GR_Data gr = new GR_Data(line, fs, this.log);
            Preconditions.checkArgument((gr.mMax > gr.mMin ? 1 : 0) != 0, (Object)"GR b=0 branch can't handle floating CH (mMin=mMax)");
            grData.add(gr);
        }
        for (GR_Data gr : grData) {
            gr.weight *= 0.5;
            this.grBuilder(gr, md, fs);
            double tmr = NSHMP_Utils.totalMoRate(gr.mMin, gr.nMag, gr.dMag, gr.aVal, gr.bVal);
            double tsm = NSHMP_Utils.totalMoRate(gr.mMin, gr.nMag, gr.dMag, 0.0, 0.0);
            gr.aVal = Math.log10(tmr / tsm);
            gr.bVal = 0.0;
            this.grBuilder(gr, md, fs);
        }
    }

    private void chBuilder(CH_Data ch, MagData md, FaultSource fs) {
        double tmr = ch.rate * MagUtils.magToMoment(ch.mag);
        double tcr = ch.rate;
        if (md.hasEpistemic) {
            for (int i = 0; i < md.numEpiBranches; ++i) {
                IncrementalMagFreqDist mfd;
                double epiMag = ch.mag + md.epiDeltas[i];
                if (md.hasAleatory) {
                    mfd = new GaussianMagFreqDist(md.aleaMinMag(epiMag), md.aleaMaxMag(epiMag), md.aleaMagCt);
                    double mfdWeight = ch.weight * md.epiWeights[i];
                    if (md.momentBalance) {
                        ((GaussianMagFreqDist)mfd).setAllButCumRate(epiMag, md.aleaSigma, tmr * mfdWeight, 2.0, 2);
                    } else {
                        ((GaussianMagFreqDist)mfd).setAllButTotMoRate(epiMag, md.aleaSigma, tcr * mfdWeight, 2.0, 2);
                    }
                    fs.mfds.add(mfd);
                    if (!this.log.isLoggable(Level.FINE)) continue;
                    this.log.fine("CH MFD [+epi +ale], M-branch " + (i + 1) + ": " + fs.name + IOUtils.LINE_SEPARATOR + mfd.getMetadataString());
                    continue;
                }
                mfd = new IncrementalMagFreqDist(epiMag, epiMag, 1);
                double tmpMoRate = tmr * ch.weight * md.epiWeights[i];
                double tmpRate = tmpMoRate / MagUtils.magToMoment(epiMag);
                mfd.set(epiMag, tmpRate);
                fs.mfds.add(mfd);
                if (!this.log.isLoggable(Level.FINE)) continue;
                this.log.fine(IOUtils.LINE_SEPARATOR + "CH MFD [+epi -ale], M-branch " + (i + 1) + ": " + fs.name + IOUtils.LINE_SEPARATOR + mfd.getMetadataString());
            }
        } else if (md.hasAleatory) {
            GaussianMagFreqDist mfd = new GaussianMagFreqDist(md.aleaMinMag(ch.mag), md.aleaMaxMag(ch.mag), md.aleaMagCt);
            if (md.momentBalance) {
                mfd.setAllButCumRate(ch.mag, md.aleaSigma, tmr * ch.weight, 2.0, 2);
            } else {
                mfd.setAllButTotMoRate(ch.mag, md.aleaSigma, tcr * ch.weight, 2.0, 2);
            }
            fs.mfds.add(mfd);
            if (this.log.isLoggable(Level.FINE)) {
                this.log.fine("CH MFD [-epi +ale]: " + fs.name + IOUtils.LINE_SEPARATOR + mfd.getMetadataString());
            }
        } else {
            IncrementalMagFreqDist mfd = new IncrementalMagFreqDist(ch.mag, ch.mag, 1);
            mfd.set(ch.mag, ch.weight * ch.rate);
            fs.mfds.add(mfd);
            if (this.log.isLoggable(Level.FINE)) {
                this.log.fine(IOUtils.LINE_SEPARATOR + "CH MFD [-epi -ale]: " + fs.name + IOUtils.LINE_SEPARATOR + mfd.getMetadataString());
            }
        }
    }

    private void grBuilder(GR_Data gr, MagData md, FaultSource fs) {
        double tmr = NSHMP_Utils.totalMoRate(gr.mMin, gr.nMag, gr.dMag, gr.aVal, gr.bVal);
        if (md.hasEpistemic) {
            double mMax = gr.mMax;
            double weight = gr.weight;
            for (int i = 0; i < md.numEpiBranches; ++i) {
                gr.mMax = mMax + md.epiDeltas[i];
                gr.updateMagCount();
                if (gr.nMag > 0) {
                    gr.weight = weight * md.epiWeights[i];
                    GutenbergRichterMagFreqDist mfd = this.makeGR(gr, tmr);
                    fs.mfds.add(mfd);
                    if (!this.log.isLoggable(Level.FINE)) continue;
                    this.log.fine(IOUtils.LINE_SEPARATOR + "GR MFD, M-branch " + (i + 1) + ": " + fs.name + IOUtils.LINE_SEPARATOR + mfd.getMetadataString());
                    continue;
                }
                StringBuilder sb = new StringBuilder().append("GR MFD epi branch with no mags");
                FaultParser.appendFaultDat(sb, fs);
                this.log.warning(sb.toString());
            }
        } else {
            GutenbergRichterMagFreqDist mfd = this.makeGR(gr, tmr);
            fs.mfds.add(mfd);
            if (this.log.isLoggable(Level.FINE)) {
                this.log.fine(IOUtils.LINE_SEPARATOR + "GR MFD: " + IOUtils.LINE_SEPARATOR + mfd.getMetadataString());
            }
        }
    }

    private GutenbergRichterMagFreqDist makeGR(GR_Data gr, double totMoRate) {
        GutenbergRichterMagFreqDist mfd = new GutenbergRichterMagFreqDist(gr.mMin, gr.nMag, gr.dMag);
        mfd.setAllButTotCumRate(gr.mMin, gr.mMin + (double)(gr.nMag - 1) * gr.dMag, gr.weight * totMoRate, gr.bVal);
        return mfd;
    }

    static void generateFaultTrace(Iterator<String> it, FaultSource fs, Logger log) {
        FaultParser.readFaultGeom(it.next(), fs);
        int traceCount = NSHMP_Utils.readInt(it.next(), 0);
        List<String> traceDat = NSHMP_Utils.readLines(it, traceCount);
        FaultTrace trace = new FaultTrace(fs.name);
        for (String ptDat : traceDat) {
            String[] latlon = StringUtils.split((String)ptDat);
            trace.add(new Location(NSHMP_Utils.readDouble(latlon, 0), NSHMP_Utils.readDouble(latlon, 1), fs.top));
        }
        if (fs.dip < 0.0) {
            fs.dip = -fs.dip;
            trace.reverse();
        }
        fs.trace = trace;
        if (log.isLoggable(Level.FINE)) {
            log.fine(IOUtils.LINE_SEPARATOR + trace.toString());
        }
    }

    private FaultSource createSource(String src, SourceFile file, int nameIdx) {
        FaultSource fs = new FaultSource();
        fs.file = file;
        Object[] fltDat = StringUtils.split((String)src);
        fs.type = FaultType.typeForID(NSHMP_Utils.readInt((String[])fltDat, 0));
        fs.mech = FocalMech.typeForID(NSHMP_Utils.readInt((String[])fltDat, 1));
        fs.nMag = NSHMP_Utils.readInt((String[])fltDat, 2);
        fs.name = StringUtils.join((Object[])fltDat, (char)' ', (int)nameIdx, (int)fltDat.length);
        fs.mfds = Lists.newArrayList();
        return fs;
    }

    private static void readFaultGeom(String line, FaultSource fs) {
        String[] fltDat = StringUtils.split((String)line);
        fs.dip = NSHMP_Utils.readDouble(fltDat, 0);
        fs.width = NSHMP_Utils.readDouble(fltDat, 1);
        fs.top = NSHMP_Utils.readDouble(fltDat, 2);
    }

    static void appendFaultDat(StringBuilder b, FaultSource fs) {
        b.append(IOUtils.LINE_SEPARATOR).append("          ").append(fs.name).append(IOUtils.LINE_SEPARATOR).append("          ").append(fs.file);
    }

    static void toLog(Logger log, FaultSource fs) {
        if (log.isLoggable(Level.INFO)) {
            log.info(IOUtils.LINE_SEPARATOR + fs.toString());
        }
    }

    private void cleanStrikeSlip(List<FaultSource> list) {
        Collection ssSrcs = Collections2.filter(list, SourcePredicates.mech(FocalMech.STRIKE_SLIP));
        HashMap<String, FaultSource> cleanSrcs = new HashMap<String, FaultSource>();
        for (FaultSource fs : ssSrcs) {
            if (!cleanSrcs.containsKey(fs.name)) {
                cleanSrcs.put(fs.name, fs);
                continue;
            }
            ((FaultSource)cleanSrcs.get((Object)fs.name)).mfds.addAll(fs.mfds);
        }
        list.removeAll(ssSrcs);
        list.addAll(cleanSrcs.values());
    }

    static void skipHeader1(Iterator<String> it) {
        int numSta = NSHMP_Utils.readInt(it.next(), 0);
        Iterators.advance(it, (int)(numSta > 0 ? numSta : 2));
        it.next();
    }

    static void skipHeader2(Iterator<String> it) {
        int nP = NSHMP_Utils.readInt(it.next(), 0);
        for (int i = 0; i < nP; ++i) {
            double epi = NSHMP_Utils.readDouble(it.next(), 1);
            if (epi > 0.0) {
                Iterators.advance(it, (int)3);
            }
            it.next();
            it.next();
            it.next();
            int nAR = NSHMP_Utils.readInt(it.next(), 0);
            Iterators.advance(it, (int)nAR);
        }
        it.next();
    }

    public static void main(String[] args) {
        Logger log = NSHMP_Utils.logger();
        Level level = Level.ALL;
        log.setLevel(level);
        for (Handler h : NSHMP_Utils.logger().getHandlers()) {
            h.setLevel(level);
        }
        log.info(String.valueOf(new Date()) + " " + FaultParser.class.getName());
        FaultParser dev = new FaultParser(log);
        SourceFile sf = SourceMgr.get(SourceRegion.WUS, SourceType.FAULT, "brange.3dip.gr.in").get(0);
        FaultERF erf = dev.parse(sf);
    }
}

