package org.scec.useit.forecasting;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.opensha.sha.simulators.RSQSimEvent;

import scratch.UCERF3.enumTreeBranches.DeformationModels;
import scratch.UCERF3.enumTreeBranches.FaultModels;
import scratch.kevin.simulators.RSQSimCatalog;
import scratch.kevin.simulators.RSQSimCatalog.Loader;

public class CatalogLoadDemo {

	public static void main(String[] args) throws IOException {
		// input catalog dir. At a minimum, this directory must contain the following 6 files
		// 		<prefix>.dList
		// 		<prefix>.eList
		// 		<prefix>.tList
		// 		<prefix>.pList
		// 		<parameter file>.in
		// 		<fault file>.flt or <fault fault>.in
		// 
		// The prefix for the 4 list files can be empty, so those files
		// could be called simply .pList, .eList, etc...
		// 
		// There is no convention for the name of the parameter file, except that is usually ends in ".in,"
		// though there can be multiple ".in" files and you'll need to make sure that the correct one is there.
		//
		// Similar for the fault input file. It's name will be listed in the 'faultFname' field in the parameter
		// file. It will typically end with ".in" or ".flt"
		File catalogDir = new File("/data/kevin/simulators/catalogs/rundir2585_1myr");
		// on windows, this would look something like: new File("C:\\Users\\intern\\<sub-dirs>...\\<catalog-dir>"

		RSQSimCatalog catalog = new RSQSimCatalog(catalogDir, "Catalog Name",
				FaultModels.FM3_1, DeformationModels.GEOLOGIC);
		// UCERF3 fault and deformation model used as input to the simulation
		// so far all have been FM 3.1/GEOLOGIC, you likely won't change these
		
		// now load the events using catalog.loader()
		Loader loader = catalog.loader();

		// you'll almost always want to filter them as they are loaded to reduce memory requirements.
		// set this as high as you can to do what you need
		loader.minMag(6.5);

		// we usually skip the first 5k years to avoid spin up time
		loader.skipYears(5000);
		
		// now read in the events, doing any calculations
		int eventCount = 0;
		double startTime = Double.NaN;
		double endTime = Double.NaN;
		for (RSQSimEvent event : loader.iterable()) {
			double timeYears = event.getTimeInYears();
			if (eventCount == 0)
				startTime = timeYears;
			endTime = timeYears;
			double mag = event.getMagnitude();
			eventCount++;
		}
		System.out.println("Loaded "+eventCount+" events");
		double duration = endTime - startTime;
		System.out.println("Duration: "+duration+" years");
		
		// if possible, it's best to do everything in one loop, so that you don't have to store
		// the entire catalog in memory (or read it from disk multiple times).
		// if you must process things by loading the whole catalog into memory at once, do this instead:
//		List<RSQSimEvent> events = loader.load();
//		for (RSQSimEvent event : events) {
//			// do stuff
//		}
		// if you do it this way, and you see an exception related to "OutOfMemory" or "heap space",
		// then google "out of memory eclipse java heap space" and follow the instructions to increase
		// total memory available to the program. Set it as high as possible, if you have 16 GB on your
		// machine then set it to 14 GB.
	}

}
